/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: AbstractFSObject
    Created: 2 January, 2003
        $Id: AbstractFSObject.java 169 2009-11-07 17:44:38Z dirk $
  $Revision: 169 $
      $Date: 2009-11-07 18:44:38 +0100 (Sa, 07 Nov 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir;

import java.io.File;
import java.util.Calendar;
import java.util.HashMap;
import java.util.StringTokenizer;

/**
 * Abstract base class for objects representing directories and files in the
 * file system.
 */
public abstract class AbstractFSObject
{
    public static final String TITLE       = "title";
    public static final String SUBTITLE    = "subtitle";
    public static final String HREF        = "href";
    public static final String DATE        = "date";
    public static final String DATE_BEGIN  = "date-begin";
    public static final String DATE_END    = "date-end";
    public static final String LOCATION    = "location";
    public static final String DESCRIPTION = "description";
    public static final String REMARK      = "remark";

    public static final String REFERENCE   = "reference:";

    protected AbstractFSObject        m_parent;
    protected HashMap<String, String> m_properties;

    protected String                  m_fileName;
    protected String                  m_fileNamePart;
    protected boolean                 m_isReference;
    protected Calendar                m_lastModified;

    protected boolean                 m_bToExport;
    protected int                     m_jpegQuality;

    /**
     * Constructor of a <tt>AbstractFSObject</tt> instance.
     * @param parentObj - Parent file system entry or <tt>null</tt> for a series
     * directory entry.
     */
    public AbstractFSObject (AbstractFSObject parentObj)
    {
        m_parent = parentObj;
        m_properties = new HashMap<String, String>(10);
        m_fileName = m_fileNamePart = "";
        m_isReference = false;
        m_lastModified = null;
        m_bToExport = true;
        m_jpegQuality = 0;
    } // AbstractFSObject

    /**
     * Gets the parent entry.
     * @return Parent file system entry or <tt>null</tt> for a series directory
     * that doesw not have a parent
     */
    public AbstractFSObject getParent ()
    {
        return m_parent;
    } // getParent

    /**
     * Tests whether this instance is empty and can safely be deleted.
     * @return <tt>True</tt> iff empty
     */
    public boolean isEmpty()
    {
        if (!this.isReference())
            return false;
        if (this.getTitle(false).length() == 0 &&
            this.get(DESCRIPTION).length() == 0 &&
            this.get(AbstractFSObject.SUBTITLE).length() == 0 &&
            this.get(AbstractFSObject.HREF).length() == 0)
        {
            return true;
        }
        return false;
    } // toString

    /**
     * Tests whether this instance is a reference.
     * @return <tt>True</tt> iff this instance is a reference
     */
    public boolean isReference ()
    {
        return m_isReference;
    } // isReference

    /**
     * Makes this instance a reference or real file.
     * @param bIsRef - <tt>True</tt> to make this instance a reference,
     * <tt>false</tt> to make it an ordinary file
     */
    public void setReference (boolean bIsRef)
    {
        m_isReference = bIsRef;
        if (m_isReference)
        {
            this.setFileNamePart(REFERENCE);
            this.setFileName("", false);
        }
        else
            this.setFileNamePart("");
    } // setReference

    /**
     * Gets the full path of this entry.
     * @return Full path including the path to the series directory
     */
    public String getFullPath ()
    {
        String name = m_fileName;

        if (name.length() == 0)
            return "";
        if (this.getParent() != null)
            return this.getParent().getFullPath() + File.separator + name;
        return name;
    } // getFullPath

    /**
     * Gets the path relative to the series directory entry.
     * @param bIncludeSeriesName - <tt>True</tt> to include the file name of the
     * series directory into the path. If <tt>false</tt>, the path starts with
     * the immediate subdirectory of the series directory.
     * @return String denoting the path
     */
    public String getPath (boolean bIncludeSeriesName)
    {
        if (m_fileName.equals("") && this.getParent() == null)
            return "";
        if (this.getParent() != null)
        {
            String path = this.getParent().getPath(bIncludeSeriesName);
            if (path != null && path.length() > 0)
                path += File.separator;
            return path + m_fileName;
        }
        return (bIncludeSeriesName) ? this.getFileName() : "";
    } // getPath

    /**
     * Gets the modification date/time of this entry.
     * @return Calendar object denoting the file modification date/time or
     * <tt>null</tt> if it is not available.
     */
    public Calendar getModDateTime ()
    {
        File f;
        long modTime;

        if (m_lastModified != null)
            return m_lastModified;
        if (this.getFullPath().length() == 0)
            return null;
        f = new File(this.getFullPath());
        if (!f.exists())
            return null;
        modTime = f.lastModified();
        if (modTime == 0L)
            return null;
        m_lastModified = Calendar.getInstance();
        m_lastModified.setTimeInMillis(modTime);
        return m_lastModified;
    } // getModDateTime

    private static final String[] monthNames =
    {
        "January", "February", "March", "April", "May", "June", "July",
        "August", "September", "October", "November", "December"
    };

    /**
     * Gets the file modification date/time as string.
     * @param bShort - If <tt>true</tt>, use the short format.
     * @return File modification date/time as string.
     */
    public String getModDateTimeString
        ( boolean bShort
        )
    {
        Calendar cal = this.getModDateTime();
        int      min, hour, day, month, year;
        String   str;

        if (cal == null)
            return "";
        min = cal.get(Calendar.MINUTE);
        hour = cal.get(Calendar.HOUR_OF_DAY);
        day = cal.get(Calendar.DAY_OF_MONTH);
        month = cal.get(Calendar.MONTH) + 1;
        year = cal.get(Calendar.YEAR);
        // Format the time.
        str = " (" + ((hour <= 9) ? "0" : "") + hour + ":"
            + ((min <= 9) ? "0" : "") + min + ")";
        // Format the date.
        if (bShort)
        {
            str = ((day <= 9) ? "0" : "") + day + "."
                + ((month <= 9) ? "0" : "") + month + "." + year + str;
            return str;
        }
        return day + "  " + monthNames[month-1] + ", " + year + str;
    } // getModDateTimeString

    /**
     * Expands a date string to human-readable format.
     * @param dateString - Date in the form YYYY[-MM[-DD[-HH[-MM[-SS]]]]]
     * @param bShort - <tt>True</tt> for a short form, <tt>false</tt> for a long
     * form with spelled out month names
     * @return Human-readable date string
     */
    public String expandDate (String dateString, boolean bShort)
    {
        StringTokenizer tokens;
        int             sec, min, hour, day, month, year;
        String          str;

        if (dateString.length() == 0)
            return "";
        sec = min = hour = day = month = year = 0;
        tokens = new StringTokenizer(dateString, "-");
        try
        {
            if (tokens.hasMoreTokens())
                year = Integer.parseInt(tokens.nextToken());
            if (tokens.hasMoreTokens())
                month = Integer.parseInt(tokens.nextToken());
            if (tokens.hasMoreTokens())
                day = Integer.parseInt(tokens.nextToken());
            if (tokens.hasMoreTokens())
                hour = Integer.parseInt(tokens.nextToken());
            if (tokens.hasMoreTokens())
                min = Integer.parseInt(tokens.nextToken());
            if (tokens.hasMoreTokens())
                sec = Integer.parseInt(tokens.nextToken());
        }
        catch (Exception e)
        {
            return dateString;
        }
        str = "";
        if (bShort)
        {
            if (day > 0)
                str += ((day <= 9) ? "0" : "") + day + ".";
            if (month > 0)
                str += ((month <= 9) ? "0" : "") + month + ".";
            if (year > 0)
                str += year;
            return str;
        }
        // Long date with time.
        if (day > 0)
            str += day + " ";
        if (month > 0)
            str += monthNames[month-1];
        if (day > 0)
            str += ", ";
        else
            str += " ";
        if (year > 0)
            str += year;
        if (hour > 0)
        {
            str += " (" + ((hour <= 9) ? "0" : "") + hour + ":"
                + ((min <= 9) ? "0" : "") + min;
            if (sec > 0)
                str += ":" + ((sec <= 9) ? "0" : "") + sec;
            str += ")";
        }
        return str;
    } // expandDate

    /**
     * Gets a string of the file size.
     * @param bInKB - <tt>True</tt> to get the size in KB, <tt>false</tt> to
     * get it in bytes
     * @return The string or <tt>null</tt> for directories
     */
    public String getFileSizeStr (boolean bInKB)
    {
        return null;
    } // getFileSizeStr

    /**
     * Gets the begin date of this instance.
     * @param bExpanded - <tt>True</tt> for a human-readable string,
     * <tt>false</tt> for the value as is (YYYY-MM-DD-HH-MM-SS or part).
     * @return Formatted begin date as string
     */
    public String getBeginDate (boolean bExpanded)
    {
        String dateString = this.get(DATE_BEGIN);
        if ((dateString == null || dateString.length() == 0) &&
        	(this instanceof DirectoryObject) &&
        	this.getFileName() != null &&
        	this.getFileName().matches("^[0-9]{4}[-_ ].*"))
        {
        	// Use the year at the beginning of the file name.
        	dateString = this.getFileName().substring(0, 4);
        	this.set(DATE_BEGIN, dateString);
        }
        if (!bExpanded)
            return dateString;
        return this.expandDate(dateString, false);
    } // getBeginDate

    /**
     * Gets the year of this file system object.
     * @return The year or <tt>null</tt> if unknown
     */
    public Integer getYear ()
    {
    	String          str;
        StringTokenizer tok;
        Integer         val;

        val = null;
        str = this.getBeginDate(false);
        if (str != null && str.length() > 0)
        {
	        tok = new StringTokenizer(str, "-");
	        if (tok.hasMoreTokens())
	            val = new Integer(tok.nextToken().trim());
        }
    	if (val == null && this.getParent() != null)
    		return this.getParent().getYear();
    	return val;
    } // getYear

    /**
     * Gets the begin date of this instance.
     * @param bExpanded - <tt>True</tt> for a human-readable string,
     * <tt>false</tt> for the value as is (YYYY-MM-DD-HH-MM-SS or part).
     * @return Formatted end date as string
     */
    public String getEndDate (boolean bExpanded)
    {
        String dateString = this.get(DATE_END);
        if (!bExpanded)
            return dateString;
        return this.expandDate(dateString, false);
    } // getEndDate

    /**
     * Gets the file name of this entry.
     * @return File name (without path)
     */
    public String getFileName ()
    {
        return m_fileName;
    } // getFileName

    /**
     * Sets the file name of this file system entry.
     * @param name - New name
     * @param bRenameFile - <tt>True</tt> if the underlying file object should
     * be renamed, too
     * @return <tt>Boolean</tt> indicating the success of this operation
     */
    public boolean setFileName (String name, boolean bRenameFile)
    {
        File oldFile = null;
        if (name == null)
            name = "";
        name = name.trim();
        if (bRenameFile && name.length() > 0 &&
            !m_fileName.equalsIgnoreCase(name))
        {
            // Remember the old file name...
            oldFile = new File(this.getFullPath());
            if (!oldFile.exists())
                oldFile = null;
        }
        if (m_parent != null)
            m_fileName = name;
        else
        {
            // Series directory - we must keep the path.
            String path = m_fileName;
            if (File.separatorChar != '/')
                path.replace('/', File.separatorChar);
            int index = path.lastIndexOf(File.separatorChar);
            if (index >= 0)
                path = path.substring(0, index);
            if (path.length() > 0)
                path += File.separator;
            m_fileName = path + name;
        }
        m_lastModified = null;
        if (oldFile != null)
            return oldFile.renameTo(new File(this.getFullPath()));
        return true;
    } // setFileName

    /**
     * Gets the file name part of this entry.
     * @return File name part from the metadata file
     */
    public String getFileNamePart ()
    {
        return m_fileNamePart;
    } // getFileNamePart

    /**
     * Sets the file name part of this file system entry.
     * @param name - New name part
     */
    public void setFileNamePart (String name)
    {
        m_fileNamePart = name;
    } // setFileNamePart

    /**
     * Gets or guesses the file or directory title.
     * @param bGuess - If <tt>true</tt>, the title is guessed from the file
     * name by applying delicious regex substitutions.
     * @return Title as string
     */
    public String getTitle (boolean bGuess)
    {
        String title, str;

        title = this.get(TITLE);
        if (title.length() > 0)
            return title;
        if (!bGuess)
            return "";
        title = this.getFileName();
        if (title.length() == 0)
            return "";
        // Remove a series and file number at the beginning.
        title = title.replaceFirst("^[0-9]+[A-Z][-_ ]", "");
        title = title.replaceFirst("^[0-9]+[-_ ]", "");
        // Replace dashes and underscores.
        title = title.replace('_', ' ');
        title = title.replace('-', ' ');
        try
        {
            if (!(this instanceof DirectoryObject))
            {
                // Remove the extension from the file name.
                title = title.replaceFirst("\\.[A-Za-z0-9_]+$", "");
            }
            // Remove '%20'.
            title = title.replaceAll("%[0-9a-fA-F]{2}", " ");
            // Expand occurrences like "TestString" to "Test String". The
            // regexp contains "[a-z]{3,}" to prevent abbreviations from
            // being affected.
            title = title.replaceAll("([a-z]{3,})([A-Z][a-z])", "$1 $2");
            // Capitalize each longer word.
            StringTokenizer tokens = new StringTokenizer(title, " ");
            title = "";
            while (tokens.hasMoreTokens())
            {
                str = tokens.nextToken();
                if (title.length() == 0 && str.equalsIgnoreCase("x"))
                    continue;
                if (title.length() == 0 || str.length() > 3)
                    str = str.substring(0,1).toUpperCase() + str.substring(1);
                if (title.length() > 0)
                    title += " ";
                title += str;
            }
        }
        catch (Exception e)
        {
            System.err.println("E: GetTitle - Regular expression exception:\n"
                + e.toString());
        }
        // Return the title string.
        return title;
    } // getTitle

    /**
     * Gets the title with all HTML tags and entities removed.
     * @return Title with all HTML tags removed
     */
    public String getTitlePlain ()
    {
        String str = this.getTitle(true);

        // Remove the HTML tags from the title.
        str = str.replaceAll("</?[A-Za-z][A-Za-z0-9_:-]* *[^>]*>", "");
        // Replace known entity references.
        str = str.replaceAll("&quot;", "\"");
        str = str.replaceAll("&lt;", "<");
        str = str.replaceAll("&gt;", ">");
        str = str.replaceAll("&nbsp;", " ");
        str = str.replaceAll("&#15.;", "-");
        str = transformAccents(str, false);
        // Replace unknown entity references and white space.
        str = str.replaceAll("&([A-Za-z])[A-Za-z0-9]*;", "$1");
        str = str.replaceAll("\n+", " ");
        str = str.replaceAll("  +", " ");
        str = str.trim();
        return str;
    } // getTitlePlain

    private static final String[] CHAR_TAB = {
        "", "&Auml;", "", "&Ouml;", "", "&Uuml;",
        "", "&auml;", "", "&iuml;", "", "&ouml;", "", "&uuml;", "", "&yuml;", "", "&szlig;",
        "", "&Aacute;", "", "&Eacute;", "", "&Iacute;", "", "&Oacute;", "", "&Uacute;",
        "", "&aacute;", "", "&eacute;", "", "&iacute;", "", "&oacute;", "", "&uacute;",
        "", "&Agrave;", "", "&Egrave;", "", "&Igrave;", "", "&Ograve;", "", "&Ugrave;",
        "", "&agrave;", "", "&egrave;", "", "&igrave;", "", "&ograve;", "", "&ugrave;",
        "", "&Acirc;", "", "&Ecirc;", "", "&Icirc;", "", "&Ocirc;", "", "&Ucirc;",
        "", "&acirc;", "", "&ecirc;", "", "&icirc;", "", "&ocirc;", "", "&ucirc;",
        "", "&Atilde;", "", "&Ntilde;", "", "&Otilde;",
        "", "&atilde;", "", "&ntilde;", "", "&otilde;",
        "", "&Ccedil;", "", "&ccedil;"
    };

    /**
     * Private method to transform accented characters to their HTML equivalent
     * and reverse.
     * @param str - The string to transform
     * @param bToHTML - <tt>True</tt> to transform to HTML, <tt>false</tt> to
     * transform to accented characters
     * @return The transformed string
     */
    public static String transformAccents (String str, boolean bToHTML)
    {
        if (str == null)
            return "";
        for (int i = 0; i < CHAR_TAB.length; i += 2)
        {
            if (bToHTML)
                str = str.replaceAll(CHAR_TAB[i], CHAR_TAB[i+1]);
            else
                str = str.replaceAll(CHAR_TAB[i+1], CHAR_TAB[i]);
        }
        return str;
    } // transformAccents

    /**
     * Checks whether this instance can be exported to the Web directory.
     * @return <tt>True</tt> iff this instance should be exported
     */
    public boolean isToExport ()
    {
        if (this.getParent() != null && !this.getParent().isToExport())
            return false;
        return m_bToExport;
    } // isToExport

    /**
     * Sets whether this instance can be exported to the Web directory.
     * @param bToExport - <tt>True</tt> to export this object
     */
    public void setToExport (boolean bToExport)
    {
        m_bToExport = bToExport;
    } // setToExport

    /**
     * Gets the conversion quality.
     * @return Conversion quality: 0: default, or 1 - 99
     */
    public int getConversionQuality ()
    {
        int parentCQ = 0;
        if (this.getParent() != null)
            parentCQ = this.getParent().getConversionQuality();
        if (parentCQ != 0 && m_jpegQuality == 0)
            return parentCQ;
        return m_jpegQuality;
    } // getConversionQuality

    /**
     * Sets the conversion quality.
     * @param quality - Conversion quality: 0: default, or 1 - 99
     */
    public void setConversionQuality (int quality)
    {
        m_jpegQuality = quality;
    } // setConversionQuality

    /**
     * Sets the value of a named property in this instance.
     * @param name - Property name
     * @param value - Property value as string
     */
    public void set (String name, String value)
    {
        name = name.toLowerCase();
        if (name.equals(DATE))
        {
            StringTokenizer tokens = new StringTokenizer(value, "/");
            if (tokens.hasMoreTokens())
                this.set(DATE_BEGIN, tokens.nextToken());
            if (tokens.hasMoreTokens())
                this.set(DATE_END, tokens.nextToken());
        }
        else
            m_properties.put(name, value);
    } // set

    /**
     * Gets the additional property names used in this instance.
     * @return String array containing the property names used by this
     * instance
     */
    public String[] getUsedPropertyNames ()
    {
        return m_properties.keySet().toArray(new String[0]);
    } // getUsedPropertyNames

    /**
     * Gets a named property of this instance.
     * @param name - Property name
     * @return Property value as string or <tt>null</tt> if there is no
     * property value for this name
     */
    public String get (String name)
    {
        if (name == DATE)
        {
            return this.getBeginDate(false) + "/"
                + this.getEndDate(false);
        }
        Object val = m_properties.get(name);
        return (val != null) ? ((String) val).trim() : "";
    } // get

    /**
     * Gets a named property of this instance.
     * @param name - Property name
     * @param bToHTML - <tt>True</tt> to transform to HTML, <tt>false</tt> to
     * transform to accented characters
     * @return Property value as string or <tt>null</tt> if there is no
     * property value for this name
     */
    public String getTransformed (String name, boolean bToHTML)
    {
        return transformAccents(this.get(name), bToHTML);
    } // getTransformed

    /**
     * Removes an additional property from this instance.
     * @param name - Property name to be removed
     */
    public void remove (String name)
    {
        m_properties.remove(name);
    } // remove

    /**
     * Returns a string representation of this instance.
     * @return The string representation
     */
    @Override
	public String toString ()
    {
        // This removes the path in series directories.
        String fileName = this.getFileName();
        if (fileName.length() > 0)
            return fileName;
        String s = this.getTransformed(TITLE, false);
        if (s.length() > 0)
            return s;
        if (m_fileNamePart.length() > 0)
            return m_fileNamePart;
        return "";
    } // toString
} // AbstractFSObject
