/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: DirPersistDecorator
    Created: 04.02.2006 (10:04:52)
        $Id: DirPersistDecorator.java 52 2006-02-10 18:57:12Z Dirk $
  $Revision: 52 $
      $Date: 2006-02-10 19:57:12 +0100 (Fr, 10 Feb 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir.persist;

import java.io.File;

import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.settings.Settings;

/**
 * Acts as a decorator for the directory persistens. It chooses a suitable
 * persister on demand. Instances of this class are returned by the
 * {@link PersistFactory}.
 * @author Dirk Grossmann
 */
public class DirPersistDecorator implements IDirPersist
{
    public static boolean s_bSaveAsXML      = true;
    public static boolean s_bExportAsXML    = true;
    public static String  s_saveXmlEncoding = Settings.METADEFAULT_XML_ENCODING;
    public static String  s_exportXmlEncoding =
        Settings.METADEFAULT_XML_ENCODING;

    /**
     * Loads the persistence-relevant settings on start-up.
     * @param settings - The settings object.
     */
    public static void loadSettings (Settings settings)
    {
        String enc;

        s_bSaveAsXML = settings.getBoolean(Settings.METADATA_SAVE_XML, true);
        s_bExportAsXML = settings.getBoolean(Settings.METADATA_EXPORT_XML,true);
        enc = settings.get(Settings.METADATA_SAVE_ENCODING);
        s_saveXmlEncoding = (enc.length() > 0) ? enc :
            Settings.METADEFAULT_XML_ENCODING;
        enc = settings.get(Settings.METADATA_EXPORT_ENCODING);
        s_exportXmlEncoding = (enc.length() > 0) ? enc :
            Settings.METADEFAULT_XML_ENCODING;
    } // loadSettings

    private Settings           m_settings;
    private AbstractDirPersist m_primaryPersister, m_primaryWebPersister;
    private AbstractDirPersist m_secondaryPersister, m_secondaryWebPersister;

    /**
     * Creates a new <tt>DirPersistDecorator</tt> instance.
     * @param settings - The Settings object
     */
    public DirPersistDecorator (Settings settings)
    {
        AbstractDirPersist filePersister;
        DirPersistXmlFile  xmlFilePersister;

        m_settings = settings;
        filePersister = new DirPersistFile(settings);
        xmlFilePersister = new DirPersistXmlFile(settings);
        xmlFilePersister.setEncodings(s_saveXmlEncoding, s_exportXmlEncoding);
        // Set the save persister preference.
        if (s_bSaveAsXML)
        {
            m_primaryPersister = xmlFilePersister;
            m_secondaryPersister = filePersister;
        }
        else
        {
            m_primaryPersister = filePersister;
            m_secondaryPersister = xmlFilePersister;
        }
        // Set the export persister preference.
        if (s_bExportAsXML)
        {
            m_primaryWebPersister = xmlFilePersister;
            m_secondaryWebPersister = filePersister;
        }
        else
        {
            m_primaryWebPersister = filePersister;
            m_secondaryWebPersister = xmlFilePersister;
        }
    } // DirPersistDecorator

    /**
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#canLoad(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public boolean canLoad (DirectoryObject seriesDirObj)
    {
        if (m_primaryPersister.canLoad(seriesDirObj))
            return true;
        return m_secondaryPersister.canLoad(seriesDirObj);
    } // canLoad

    /**
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#load(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public void load (DirectoryObject seriesDirObj) throws PersistException
    {
        if (m_primaryPersister.canLoad(seriesDirObj))
            m_primaryPersister.load(seriesDirObj);
        else
            m_secondaryPersister.load(seriesDirObj);
    } // load

    /**
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#save(com.dgrossmann.photo.dir.DirectoryObject, boolean)
     */
    public void save (DirectoryObject seriesDirObj, boolean bForce)
        throws PersistException
    {
        // Save using the primary persister.
        m_primaryPersister.save(seriesDirObj, bForce);
        // When the save was successful, we can delete the file of the other.
        File f = new File(m_secondaryPersister.getMetaDataStoreName
            (seriesDirObj, true));
        if (f.exists())
            f.delete();
    } // save

    /**
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#saveForWeb(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public void saveForWeb (DirectoryObject seriesDirObj)
        throws PersistException
    {
        m_primaryWebPersister.saveForWeb(seriesDirObj);
        // When the save was successful, we can delete the file of the other.
        File f = new File(m_settings.get(Settings.EXPORT_DIRECTORY),
            m_secondaryWebPersister.getMetaDataStoreName(seriesDirObj, false));
        if (f.exists())
            f.delete();
    } // saveForWeb
} // DirPersistDecorator
