/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ExplorerMainFrame
    Created: 2 January, 2003
        $Id: ExplorerMainFrame.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import javax.help.CSH;
import javax.help.HelpBroker;
import javax.help.HelpSet;
import javax.swing.*;
import javax.swing.tree.TreePath;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.dir.persist.PersistException;
import com.dgrossmann.photo.dir.persist.PersistFactory;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.dialog.AboutDialog;
import com.dgrossmann.photo.ui.dialog.EnterDirNameDialog;
import com.dgrossmann.photo.ui.dialog.ExportProgressDialog;
import com.dgrossmann.photo.ui.dialog.ExportSelDialog;
import com.dgrossmann.photo.ui.dialog.PropertiesDialog;
import com.dgrossmann.photo.ui.dialog.SettingsDialog;
import com.dgrossmann.photo.ui.panel.DirTreePanel;
import com.dgrossmann.photo.ui.panel.IExplorerPanel;
import com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener;
import com.dgrossmann.photo.ui.panel.PropertyPanel;
import com.dgrossmann.photo.ui.panel.contents.ContentsPanel;
import com.dgrossmann.photo.webexport.ExportFactory;
import com.dgrossmann.photo.webexport.ExportProgress;
import com.dgrossmann.photo.webexport.ExportThread;
import com.dgrossmann.photo.webexport.IWebExport;

/**
 * The main frame window classs of the <i>J Photo-Explorer</i> application that
 * contains the implementation of all action commands.
 * @author Dirk Grossmann
 */
public class ExplorerMainFrame extends JFrame
    implements IExplorerPanelChangeListener
{
    // Application help information.
    static final String HELPSET_NAME            = "MainHelp";

    // Window Settings property names.
    static final String MAIN_WIN_POS            = "main.window.pos";
    static final String MAIN_WIN_MAXIMIZED      = "main.window.maximized";
    static final String MAIN_WIN_DIR_SPLIT_POS  = "main.window.dir_split_pos";
    static final String MAIN_WIN_PROP_SPLIT_POS = "main.window.prop_split_pos";

    private Settings               m_settings;
    private SeriesContainer        m_seriesContainer;
    private ImageHolder            m_imgHolder;

    // Cut & Paste information.
    private DirectoryObject        m_currentDir;
    private List<AbstractFSObject> m_objsToPaste;

    // Panel components.
    private DirTreePanel           m_dirTreePanel;
    private IExplorerPanel         m_propertyPanel;
    private ContentsPanel          m_dirContentsPanel;

    private JMenuItem              m_helpContentsMenuItem;
    private JMenuItem              m_aboutMenuItem;

    private HelpSet                m_mainHS;
    private HelpBroker             m_mainHB;
    private String                 m_helpError;

    // Web export information.
    public Boolean                 m_exportReady;
    public ExportProgress          m_exportProgress;
    public javax.swing.Timer       m_exportTimer;
    public String                  m_exportError;
    public boolean                 m_exportStop;

    /**
     * Creates a new <tt>ExplorerMainFrame</tt> instance that represents the
     * main window of the J-Photo Explorer application.
     */
    public ExplorerMainFrame()
    {
        m_currentDir = null;
        m_objsToPaste = new ArrayList<AbstractFSObject>(40);
        m_settings = new Settings("jphoto-explorer.properties");
        PersistFactory.loadSettings(m_settings);
        m_seriesContainer = new SeriesContainer(m_settings);
        m_imgHolder = new ImageHolder();
        m_exportReady = new Boolean(true);
        m_exportProgress = null;
        m_exportTimer = null;
        m_exportStop = false;
        // Initialize the GUI components and create the panels.
        this.initComponents();
        m_dirTreePanel = new DirTreePanel(this);
        dirPanel.add(m_dirTreePanel.getRootComponent(),
            BorderLayout.CENTER);
        m_propertyPanel = new PropertyPanel(this);
        propertyPanel.add(m_propertyPanel.getRootComponent(),
            BorderLayout.CENTER);
        m_dirContentsPanel = new ContentsPanel(this);
        contentsPanel.add(m_dirContentsPanel.getRootComponent(),
            BorderLayout.CENTER);
        // Initialize the panels.
        m_dirTreePanel.addChangeListener(this);
        m_propertyPanel.addChangeListener(this);
        m_dirContentsPanel.addChangeListener(this);
        this.setStatusText();
        // Help menu items.
        m_helpError = "";
        m_mainHS = null;
        m_mainHB = null;
        try
        {
            ClassLoader cl = this.getClass().getClassLoader();
            URL url = HelpSet.findHelpSet(cl, HELPSET_NAME);
            m_mainHS = new HelpSet(cl, url);
        }
        catch (NoClassDefFoundError nce)
        {
            m_helpError = "JavaHelp classes not found: " + nce.getMessage();
            System.err.println("E: " + m_helpError);
        }
        catch (Exception ee)
        {
            m_helpError = "Help Set \"" + HELPSET_NAME + "\" not found.";
            System.err.println("E: " + m_helpError);
        }
        catch (ExceptionInInitializerError ex)
        {
            m_helpError = "Help system initialization error:\n"
                + ex.toString();
            System.err.println("E: " + m_helpError);
            ex.getException().printStackTrace();
        }
        if (m_mainHS != null)
        {
            m_mainHB = m_mainHS.createHelpBroker();
            m_mainHB.enableHelpKey(this, "top", null);
        }
        if (m_mainHB != null)
        {
            m_helpContentsMenuItem = new JMenuItem("Help Contents");
            m_helpContentsMenuItem.addActionListener
                (new CSH.DisplayHelpFromSource(m_mainHB));
        }
        else
            m_helpContentsMenuItem = null;
        m_aboutMenuItem = new JMenuItem("About " + AppInfo.APP_NAME + " ...");
        m_aboutMenuItem.addActionListener(new ActionListener() {
            public void actionPerformed (ActionEvent evt)
            {
                ExplorerMainFrame.this.aboutMenuItemActionPerformed(evt);
            } // actionPerformed
        });
        if (m_helpContentsMenuItem != null)
        {
            helpMenu.add(m_helpContentsMenuItem);
            JSeparator sep = new JSeparator();
            sep.setBackground(new Color(255, 255, 255));
            sep.setForeground(new Color(153, 153, 153));
            helpMenu.add(sep);
        }
        helpMenu.add(m_aboutMenuItem);
        // Set the explorer frame title and icon.
        this.setTitle(AppInfo.APP_NAME);
        ImageIcon img = ImageHolder.loadStandardImage
            ("images/app-logo.gif", this);
        if (img != null)
            this.setIconImage(img.getImage());
        // Set the window frame position.
        Rectangle frameRect = m_settings.getRect(MAIN_WIN_POS);
        if (frameRect == null)
        {
            Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            frameRect = new Rectangle();
            frameRect.width = screenSize.width;
            frameRect.height = screenSize.height - 34;
            frameRect.x = screenSize.width - frameRect.width;
            frameRect.y = screenSize.height - frameRect.height - 34;
        }
        this.setBounds(frameRect);
        if (m_settings.getBoolean(MAIN_WIN_MAXIMIZED, false))
            this.setState(MAXIMIZED_BOTH);
        // Set the splitter positions.
        int dirSplitPos = m_settings.getInt(MAIN_WIN_DIR_SPLIT_POS, -1);
        int propSplitPos = m_settings.getInt(MAIN_WIN_PROP_SPLIT_POS, -1);
        if (dirSplitPos >= 0)
            dirSplitPane.setDividerLocation(dirSplitPos);
        if (propSplitPos >= 0)
            propSplitPane.setDividerLocation(propSplitPos);
        // Now make the frame window visible.
        this.setVisible(true);
        // Check the Java version.
        String javaVersion = System.getProperty("java.version");
        boolean bJavaVersionOK = false;
        StringTokenizer tok = new StringTokenizer(javaVersion, ".");
        if (tok.countTokens() >= 2)
        {
            try
            {
                float vers = Float.parseFloat
                    (tok.nextToken() + "." + tok.nextToken());
                bJavaVersionOK = (vers >= 1.4F);
            }
            catch (Exception ignored)
            {
            }
        }
        if (!bJavaVersionOK)
        {
            String javaVendor = System.getProperty("java.vendor");
            String msg;
            msg = AppInfo.APP_NAME + " needs Java Version 1.4 or higher."
                + "\n\nYou are using version " + javaVersion + " by "
                + javaVendor + "\n\nExit " + AppInfo.APP_NAME + ".";
            JOptionPane.showMessageDialog(this, msg, AppInfo.APP_NAME,
                JOptionPane.ERROR_MESSAGE);
            System.exit(1);
        }
        // Initialize the help objects.
        if (m_mainHB == null)
        {
            System.err.println("   Class Path: " +
                System.getProperty("java.class.path"));
            JOptionPane.showMessageDialog(this, "Cannot show help."
                + ((m_helpError.length() > 0) ? "\n" : "") + m_helpError
                + "\n\nClass Path: " + System.getProperty("java.class.path"),
                AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
        }
        // Show a help message if we do not have series directories.
        if (!m_seriesContainer.hasSeriesDirectories())
        {
            JOptionPane.showMessageDialog(this, AppInfo.APP_NAME
                + " does not have photo series directories. Please choose "
                + "\"Series -> Settings...\"\nto define the directories that "
                + "contain photo series directories.\n\n"
                + "Choose \"Help -> Help Contents\" for help.",
                AppInfo.APP_NAME, JOptionPane.INFORMATION_MESSAGE);
        }
        // Initialize the components for the panels after the main frame window
        // is realized.
        SwingUtilities.invokeLater(new Runnable()
        {
            public void run ()
            {
                // Allow each panel to adjust its subcomponents.
                m_dirTreePanel.setupComponents();
                m_propertyPanel.setupComponents();
                m_dirContentsPanel.setupComponents();
                // Load the panel settings.
                m_dirTreePanel.loadSettings(m_settings);
                m_propertyPanel.loadSettings(m_settings);
                m_dirContentsPanel.loadSettings(m_settings);
            } // run
        });
    } // ExplorerMainFrame

    /**
     * Gets the Explorer settings.
     * @return The settings
     */
    public Settings getSettings ()
    {
        return m_settings;
    } // getSettings

    /**
     * Returns the series container.
     * @return The series container
     */
    public SeriesContainer getSeriesContainer ()
    {
        return m_seriesContainer;
    } // getSeriesContainer

    /**
     * Gets the image holder of the Explorer frame.
     * @return The image holder
     */
    public ImageHolder getImageHolder ()
    {
        return m_imgHolder;
    } // getImageHolder

    /**
     * Gets the contents panel.
     * @return The contents panel
     */
    public IExplorerPanel getDirContentsPanel ()
    {
        return m_dirContentsPanel;
    } // getDirContentsPanel

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener#onCurrentDirectoryChanged(com.dgrossmann.photo.ui.panel.IExplorerPanel, com.dgrossmann.photo.dir.DirectoryObject)
     */
    public void onCurrentDirectoryChanged
        ( IExplorerPanel  source
        , DirectoryObject dirObj
        )
    {
        m_currentDir = dirObj;
        this.setStatusText();
        if (source == m_dirTreePanel)
        {
            m_dirContentsPanel.setCurrentDirectory(dirObj);
            m_propertyPanel.setCurrentFSObject(dirObj);
        }
        else if (source == m_dirContentsPanel)
        {
            m_dirTreePanel.setCurrentDirectory(dirObj);
            m_propertyPanel.setCurrentFSObject(dirObj);
        }
    } // onCurrentDirectoryChanged

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener#onSelectionChanged(com.dgrossmann.photo.ui.panel.IExplorerPanel, java.util.List)
     */
    public void onSelectionChanged
        ( IExplorerPanel         source
        , List<AbstractFSObject> newSelectedObjects
        )
    {
        if (source != m_propertyPanel)
            m_propertyPanel.setCurrentFSObjects(newSelectedObjects);
    } // onSelectionChanged

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener#onStructureChanged(com.dgrossmann.photo.ui.panel.IExplorerPanel, com.dgrossmann.photo.dir.DirectoryObject)
     */
    public void onStructureChanged
        ( IExplorerPanel  source
        , DirectoryObject dirObj
        )
    {
        if (source != m_dirTreePanel)
            m_dirTreePanel.onTreeStructureChanged(dirObj);
    } // onStructureChanged

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener#onPropertiesChanged(com.dgrossmann.photo.ui.panel.IExplorerPanel, com.dgrossmann.photo.dir.AbstractFSObject)
     */
    public void onPropertiesChanged
        ( IExplorerPanel   source
        , AbstractFSObject fsObj
        )
    {
        if (fsObj instanceof DirectoryObject && source != m_dirTreePanel)
            m_dirTreePanel.onTreeStructureChanged((DirectoryObject) fsObj);
        if (source != m_propertyPanel)
            m_propertyPanel.refresh();
        if (source != m_dirContentsPanel)
            m_dirContentsPanel.refresh();
    } // onPropertiesChanged

    /**
     * Creates a new file under the current directory as a copy of a selected
     * source file.
     * @param currentDir - The current directory
     * @return <tt>True</tt> on success, <tt>false</tt> if no file has been
     * created
     */
    public boolean newFile (DirectoryObject currentDir)
    {
        if (currentDir == null)
            return false;
        // Create a new file under "m_currentDir".
        JFileChooser choose = new JFileChooser();
        choose.setDialogTitle("Choose File");
        choose.setDialogType(JFileChooser.OPEN_DIALOG);
        choose.setFileSelectionMode(JFileChooser.FILES_ONLY);
        choose.setApproveButtonToolTipText
            ("Chooses the file to be copied to \""
            + currentDir.getFileName() + "\"");
        JPanel addPanel = new JPanel(new GridLayout(5, 1));
        addPanel.setBorder(new javax.swing.border.TitledBorder
            (null, "Information",
            javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION,
            javax.swing.border.TitledBorder.DEFAULT_POSITION,
            new java.awt.Font("Dialog", 0, 11),
            new java.awt.Color(51, 51, 153)));
        addPanel.add(new JLabel("The selected file"));
        addPanel.add(new JLabel("will be copied"));
        addPanel.add(new JLabel("into directory"));
        addPanel.add(new JLabel("\"" + currentDir.getFileName() + "\""));
        choose.setAccessory(addPanel);
        // Show the dialog.
        if (choose.showDialog(this,"Select") != JFileChooser.APPROVE_OPTION)
            return false;
        // Get the selected file.
        File dirFile = choose.getSelectedFile();
        String fileName = dirFile.getName();
        String newName = currentDir.getFullPath() + File.separator
            + fileName;
        if (dirFile == null || !dirFile.isFile())
            return false;
        // Copy the selected file into "m_currentDir".
        FileInputStream in = null;
        FileOutputStream out = null;
        byte[] buf = new byte[32 * 1024];
        int len;
        try
        {
            in = new FileInputStream(dirFile);
            out = new FileOutputStream(newName);
            while ((len = in.read(buf)) > 0)
                out.write(buf, 0, len);
        }
        catch (Exception e)
        {
            JOptionPane.showMessageDialog
                (this, "Cannot copy file:\n   " + dirFile + "\nto:\n   "
                + newName +e, AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
            return false;
        }
        finally
        {
            try
            {
                if (in != null)
                    in.close();
                if (out != null)
                    out.close();
            }
            catch (Exception exc)
            {
            }
        }
        // Create and add the file objects for this file.
        FileObject fe = new FileObject(fileName, currentDir);
        fe.setReference(false);
        if (!this.finishFileObj(fe))
        {
            // Properties dialog canceled - delete the file.
            dirFile = new File(newName);
            if (dirFile.exists())
                dirFile.delete();
            return false;
        }
        return true;
    } // newFile

    /**
     * Creates a new subdirectory under the current directory.
     * @param currentDir - The current directory
     * @return <tt>True</tt> on success, <tt>false</tt> if no directory has
     * been created
     */
    public boolean newDirectory (DirectoryObject currentDir)
    {
        if (currentDir == null)
            return false;
        // Create a new subfolder under "m_currentDir".
        EnterDirNameDialog dlg = new EnterDirNameDialog
            (this, currentDir, true);
        if (!dlg.showDialog())
            return false;
        // Create the new directory.
        String name = dlg.getDirName();
        File dir = new File(currentDir.getFullPath(), name);
        if (!dir.mkdir())
        {
            JOptionPane.showMessageDialog(this, "Cannot create folder \""
                + dir.getAbsolutePath() + "\"", AppInfo.APP_NAME,
                JOptionPane.ERROR_MESSAGE);
            return false;
        }
        DirectoryObject dirObj = new DirectoryObject(name, currentDir);
        return this.finishFileObj(dirObj);
    } // newDirectory

    /**
     * Completes file system object creation.
     * @param fsObj - The file system object whose creation should be completed
     * @return <tt>True</tt> iff the object has been initialized successfully
     */
    public boolean finishFileObj (AbstractFSObject fsObj)
    {
        boolean bIsSeparator = false;
        if (fsObj instanceof FileObject && ((FileObject) fsObj).isSeparator())
            bIsSeparator = true;
        if (!bIsSeparator)
        {
            PropertiesDialog dlg = new PropertiesDialog
                (m_imgHolder, fsObj, null, this, true);
            if (fsObj instanceof DirectoryObject)
                dlg.setTitle("New Folder \"" + fsObj.getFileName() + "\"");
            else if (fsObj.isReference())
                dlg.setTitle("New Reference");
            else
                dlg.setTitle("New File \"" + fsObj.getFileName() + "\"");
            if (!dlg.showDialog())
                return false;
        }
        // Add the new file system object and select it.
        if (fsObj instanceof DirectoryObject)
            m_dirTreePanel.onTreeStructureChanged(m_currentDir);
        m_dirContentsPanel.addNewFSObject(fsObj);
        return true;
    } // finishFileObj

    /**
     * Sets the status text according to the current directory or the currently
     * cut file.
     */
    public void setStatusText ()
    {
        String txt;

        // Set the window title.
        if (m_currentDir != null)
        {
            this.setTitle(m_currentDir.getTitlePlain() + " - "
                + AppInfo.APP_NAME);
        }
        else
            this.setTitle(AppInfo.APP_NAME);
        statusBar.setForeground(new Color(0, 0, 0));
        // If we have objects to paste, show them.
        if (m_objsToPaste.size() > 0)
        {
            txt = "  Cut: ";
            if (m_objsToPaste.size() == 1)
            {
                AbstractFSObject fsObj = (m_objsToPaste.get(0));
                txt += "\"" + fsObj.getTitle(true) + "\"";
                if (fsObj.isReference())
                    txt += " (Reference)";
                else
                {
                    if (fsObj instanceof DirectoryObject)
                        txt += " (Directory ";
                    else
                        txt += " (File ";
                    txt += fsObj.getFullPath() + ")";
                }
            }
            else
                txt += "" + m_objsToPaste.size() + " objects";
            statusBar.setForeground(new Color(51, 51, 153));
        }
        else if (m_currentDir != null)
        {
            // If we have a current directory, show it.
            int subdirs, files;
            if (m_currentDir.getParent() == null)
                txt = "  Series ";
            else
                txt = "  Group ";
            txt += ((m_currentDir.getParent() == null ||
                     m_currentDir.getParent().getParent() == null) ?
                    m_currentDir.getFullPath() : m_currentDir.getPath(true));
            subdirs = m_currentDir.getSubDirCount();
            files = m_currentDir.getFileCount();
            txt += " (" + subdirs +
                ((subdirs != 1) ? " subdirectories" : " subdirectory") +
                ", " + files + ((files != 1) ? " files" : " file") + ")";
        }
        else
        {
            // Otherwise, show program information.
            txt = "  " + AppInfo.APP_NAME + " Version "
                + AppInfo.getVersionString()
                + " of " + AppInfo.getBuildDate(false);
        }
        statusBar.setText(txt);
    } // setStatusText

    /**
     * Cuts the file system objects passed as parameter.
     * @param fsObjsToCut - List of the file system objects to cut
     */
    public void cut (List<AbstractFSObject> fsObjsToCut)
    {
         m_objsToPaste.clear();
         m_objsToPaste.addAll(fsObjsToCut);
         this.setStatusText();
         m_propertyPanel.refresh();
    } // cut

    /**
     * Tests whether we have objects to paste.
     * @return <tt>True</tt> iff yes
     */
    public boolean canPaste ()
    {
        return m_objsToPaste != null && m_objsToPaste.size() > 0;
    } // canPaste

    /**
     * Pasts the current objects.
     */
    public void paste ()
    {
        Iterator<AbstractFSObject>         iter;
        AbstractFSObject fsObj, p;
        DirectoryObject  oldParent = null;

        if (m_currentDir == null || m_objsToPaste.size() == 0)
            return;
        iter = m_objsToPaste.iterator();
        while (iter.hasNext())
        {
            fsObj = (iter.next());
            // If the object is a directory, we cannot paste it into one of its
            // ancestors.
            if (fsObj instanceof DirectoryObject)
            {
                p = m_currentDir;
                while (p != null)
                {
                    if (p == fsObj)
                    {
                        JOptionPane.showMessageDialog(this,
                            "Directory \"" + fsObj.getFileName()
                            + "\" cannot be moved\ninto itself or one of its "
                            + "subdirectories",
                            AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                        m_objsToPaste.clear();
                        this.setStatusText();
                        return;
                    }
                    p = p.getParent();
                }
            }
            // Paste the file object.
            this.renameExportedFiles(fsObj, m_currentDir, null);
            oldParent = (DirectoryObject) fsObj.getParent();
            if (!m_currentDir.addChild(fsObj, -1))
            {
                JOptionPane.showMessageDialog(this,
                    ((fsObj instanceof FileObject) ? "File \"" : "Directory \"")
                    + fsObj.getFileName()
                    + "\" cannot be moved.\n\nIt may be used by another "
                    + "application.", AppInfo.APP_NAME,
                    JOptionPane.ERROR_MESSAGE);
                m_objsToPaste.clear();
                this.setStatusText();
                return;
            }
        }
        // Save the old and new series (using "oldParent" is OK as all objects
        // to be pasted are within the same directory.
        try
        {
            if (oldParent != null)
                m_seriesContainer.saveSeries(oldParent, true);
            m_seriesContainer.saveSeries(m_currentDir, true);
        }
        catch (PersistException pe)
        {
            // Show the load error dialog.
            JOptionPane.showMessageDialog(this, pe.getMessage(),
                AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
        }
        // Select all file objects pasted.
        if (oldParent != null)
        {
            // Refresh the directory tree.
            m_dirTreePanel.refresh(m_currentDir);
        }
        m_dirContentsPanel.refresh();
        m_dirContentsPanel.selectCurrentFiles(false, m_objsToPaste);
        // Disable the "Paste" button. Can be removed to allow multiple paste
        // operations.
        m_objsToPaste.clear();
        this.setStatusText();
    } // paste

    /**
     * Shows the property dialog for the file system object.
     * @param fsObj - The object
     * @return <tt>True</tt> iff the dialog has been terminated with OK
     */
    public boolean showProperties (AbstractFSObject fsObj)
    {
        this.save();
        PropertiesDialog dlg = new PropertiesDialog(this.getImageHolder(),
            fsObj, null, this, true);
        return dlg.showDialog();
    } // showProperties

    /**
     * Deletes all file system objects in the list.
     * @param fileObjsToDelete - List of all file system objects to delete
     */
    public void delete (List<AbstractFSObject> fileObjsToDelete)
    {
        AbstractFSObject fsObj;
        Iterator<AbstractFSObject>         iter;
        int              res;
        boolean          bIsSeparator;
        String           beginStr, endStr;

        if (m_currentDir == null || fileObjsToDelete.size() == 0)
            return;
        iter = fileObjsToDelete.iterator();
        while (iter.hasNext())
        {
            fsObj = (iter.next());
            if (fsObj.isReference())
            {
                bIsSeparator = (fsObj instanceof FileObject &&
                    ((FileObject) fsObj).isSeparator());
                if (!bIsSeparator)
                {
                    res = JOptionPane.showConfirmDialog(this,
                        "Do you really want to delete reference \""
                        + fsObj.getTitle(true) + "\"?", AppInfo.APP_NAME,
                    JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
                    if (res != 0)
                        return;
                }
            }
            else
            {
                if (fsObj instanceof DirectoryObject)
                {
                    beginStr = "Directory \"";
                    endStr = "\nand its contents will be destroyed";
                }
                else
                {
                    beginStr = "File \"";
                    endStr = "";
                }
                res = JOptionPane.showConfirmDialog(this, beginStr
                    + fsObj.getFileName()
                    + "\" will be deleted permanently" + endStr + ".\n\n"
                    + "Do you want to continue?", AppInfo.APP_NAME,
                    JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
                if (res != 0)
                    return;
                // Delete the exportedg files.
                try
                {
                   ExportFactory.getExport
                       (m_settings, m_seriesContainer, this).
                       deleteExportedFiles(fsObj, IWebExport.EXPORT_ALL);
                }
                catch (Exception e)
                {
                }
                // Delete the underlying file object.
                if (fsObj instanceof DirectoryObject)
                    ((DirectoryObject) fsObj).delete(true);
                else
                {
                    m_imgHolder.deleteThumbnails(fsObj);
                    File f = new File(fsObj.getFullPath());
                    if (f.exists())
                        f.delete();
                    fsObj.setFileName("", false);
                }
            }
            // Delete the remaining reference.
            m_dirContentsPanel.removeFSObject(fsObj);
        }
        // Refresh the file list table.
        fileObjsToDelete.clear();
        fsObj = m_dirContentsPanel.getPreviousFile();
        if (fsObj != null)
            fileObjsToDelete.add(fsObj);
        m_dirContentsPanel.selectCurrentFiles(true, fileObjsToDelete);
        // It cannot handle pasting a deleted file.
        m_objsToPaste.clear();
        this.setStatusText();
    } // delete

    /**
     * Private helper for "startExport" to build the list of the files to be
     * exported.
     * @param dirObj
     * @param exportList
     */
    private void addExportedFiles
    	( DirectoryObject  dirObj
    	, List<FileObject> exportList
    	)
    {
    	Iterator<DirectoryObject> dirIter;
        Iterator<FileObject>      fileIter;
        FileObject                fileObj;

        // Add the files of the "dirObj" instance.
        fileIter = dirObj.getFileIterator();
        while (fileIter.hasNext())
        {
            fileObj = fileIter.next();
            if (!fileObj.isReference() && fileObj.isToExport())
                exportList.add(fileObj);
        }
        // Add the files of the sub directories.
        dirIter = dirObj.getSubDirIterator();
        while (dirIter.hasNext())
            this.addExportedFiles(dirIter.next(), exportList);
    } // addExportedFiles

    /**
     * Starts the Web export of the file system object passed as parameter.
     * @param fsObj - File system object to export
     * @param bMain - <tt>True</tt> to generate the main image size
     * @param bIndex - <tt>True</tt> to generate the index image size
     * @param bForce - <tt>True</tt> to force image generation
     */
    public void startExport
        ( AbstractFSObject fsObj
        , boolean          bMain
        , boolean          bIndex
        , boolean          bForce
        )
    {
        if (!m_exportReady.booleanValue() || m_exportTimer != null)
        {
            JOptionPane.showMessageDialog(this, "There is a Web export "
                + "operation in progress. Please try later", AppInfo.APP_NAME,
                JOptionPane.WARNING_MESSAGE);
            return;
        }
        m_imgHolder.stopBackgroundImageLoading();
        this.save();
        // Create the list of all files to be exported.
        this.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        ArrayList<FileObject> exportList = new ArrayList<FileObject>(100);
        try
        {
            if (fsObj instanceof FileObject)
            {
                m_seriesContainer.saveSeriesForWeb
                    ((DirectoryObject) fsObj.getParent());
                if (!fsObj.isReference() && fsObj.isToExport())
                {
                    m_seriesContainer.saveSeriesForWeb
                        ((DirectoryObject) fsObj.getParent());
                    exportList.add((FileObject) fsObj);
                }
            }
            else if (fsObj instanceof DirectoryObject)
            {
                DirectoryObject dirObj = (DirectoryObject) fsObj;
                if (dirObj.getParent() == null)
                    m_seriesContainer.ensureLoaded(dirObj);
                m_seriesContainer.saveSeriesForWeb(dirObj);
                this.addExportedFiles(dirObj, exportList);
            }
            else if (fsObj == null)
            {
                DirectoryObject[] series =
                    m_seriesContainer.getSeriesDirectories();
                for (int i = 0; i < series.length; i++)
                {
                    m_seriesContainer.ensureLoaded(series[i]);
                    m_seriesContainer.saveSeriesForWeb(series[i]);
                    this.addExportedFiles(series[i], exportList);
                }
            }
        }
        catch (PersistException pe)
        {
            JOptionPane.showMessageDialog(this, "Cannot export the series " +
                "metadata to the Web export directory:\n" + pe.getMessage(),
                AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
            return;
        }
        finally
        {
            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }
        if (exportList.size() == 0)
        {
            JOptionPane.showMessageDialog(this, "There are no files to export",
                AppInfo.APP_NAME, JOptionPane.INFORMATION_MESSAGE);
            return;
        }
        // Show the export progress dialog.
        ExportProgressDialog dlg = new ExportProgressDialog(this,
            ExportFactory.getExport(m_settings, m_seriesContainer, this).
            getDescription(), exportList.size());
        dlg.setVisible(true);
        m_exportProgress = null;
        m_exportError = null;
        m_exportStop = false;
        this.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        // Start the export thread and timer.
        ExportThread exThread = new ExportThread(exportList, m_settings,
            m_seriesContainer, this);
        exThread.setExportOptions(bMain, bIndex, bForce);
        exThread.start();
        m_exportTimer = new javax.swing.Timer(100,
            new ExportTimerTask(this, dlg));
        m_exportTimer.start();
    } // startExport

    /** Stops the export. */
    public void stopExport ()
    {
        m_exportStop = true;
        this.onExportReady();
    } // stopExport

    /**
     * Sets the export ready state.
     * @param state - <tt>True</tt> for ready
     */
    public void exportSetReady (boolean state)
    {
        synchronized(m_exportReady)
        {
            m_exportReady = new Boolean(state);
        }
    } // exportSetReady

    /**
     * Sets the export error state.
     * @param err - <tt>True</tt> for error
     */
    public void exportSetError (String err)
    {
        synchronized(m_exportReady)
        {
            m_exportReady = new Boolean(true);
            m_exportError = err;
        }
    } // exportSetReady

    /**
     * Gets the export progress.
     * @return The progress object
     */
    public ExportProgress getExportProgress ()
    {
        synchronized (m_exportReady)
        {
            return m_exportProgress;
        }
    } // getExportProgress

    /**
     * Sets the export progress.
     * @param p - New progress object
     */
    public void setExportProgress (ExportProgress p)
    {
        synchronized (m_exportReady)
        {
            m_exportProgress = p;
        }
    } // setExportProgress

    /**
     * Called when the export is ready.
     */
    public void onExportReady ()
    {
        m_propertyPanel.refresh();
    } // onExportReady

    /**
     * Renames the exported files for a file system object.
     * @param oldFsObj - Old file system object (before move or renaming)
     * @param newParent - New parent directory
     * @param newName - New file name
     */
    public void renameExportedFiles
        ( AbstractFSObject oldFsObj
        , DirectoryObject  newParent
        , String           newName
        )
    {
        try
        {
            // Rename the files in the export directory.
            ExportFactory.getExport
                (m_settings, m_seriesContainer, this).
                renameExportedFiles(oldFsObj, newParent, newName);
            // Rename the thumbnail files.
            if (oldFsObj instanceof FileObject)
                m_imgHolder.renameThumbnails(oldFsObj, newName);
            else
            {
                m_imgHolder.moveThumbnailDirUnder
                    ((DirectoryObject) oldFsObj, newParent);
            }
        }
        catch (Exception e)
        {
        }
    } // renameExported

    /**
     * Tries to select the directory node denoted by a tree path.
     * @param treePath - Tree path of the directory to be selected
     * @param bExact - <tt>True</tt> if an exact match is needed, <tt>false</tt>
     * if an ancestor of the node denoted by the tree path is sufficient
     * @return The actually selected directory object, or <tt>null</tt>
     * otherwise
     */
    public DirectoryObject selectDirForPath
        ( TreePath treePath
        , boolean  bExact
        )
    {
        return m_dirTreePanel.selectDirForPath(treePath, bExact);
    } // selectDirForPath

    /**
     * Refreshes all structures in the display.
     */
    public void refresh ()
    {
        this.save();
        m_objsToPaste.clear();
        m_currentDir = null;
        this.setStatusText();
        // Refresh the series structure.
        try
        {
            m_seriesContainer.refresh();
        }
        catch (Exception ignored)
        {
        }
        // Refresh the panels.
        m_dirTreePanel.refresh();
        m_dirContentsPanel.refresh();
        m_propertyPanel.refresh();
    } // refresh

    /**
     * Reloads the settings for the panels.
     */
    public void reloadSettings ()
    {
        m_dirTreePanel.loadSettings(m_settings);
        m_propertyPanel.loadSettings(m_settings);
        m_dirContentsPanel.loadSettings(m_settings);
    } // reloadSettings

    /**
     * Saves the settings for the panels.
     */
    public void saveSettings ()
    {
        m_dirTreePanel.saveSettings(m_settings);
        m_propertyPanel.saveSettings(m_settings);
        m_dirContentsPanel.saveSettings(m_settings);
    } // saveSettings

    /**
     * Reads the directory information fields and saves the current directory
     * object.
     */
    public void save ()
    {
        if (m_currentDir == null)
            return;
        // Let the panels save their changes.
        m_dirTreePanel.saveChanges();
        m_propertyPanel.saveChanges();
        m_dirContentsPanel.saveChanges();
        // Now save the complete series container.
        try
        {
            m_seriesContainer.saveSeries(m_currentDir, false);
        }
        catch (PersistException pe)
        {
            // Show the load error dialog.
            JOptionPane.showMessageDialog(this, pe.getMessage(),
                AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
        }
    } // save

    /**
     * Exits the main window and records the window position.
     */
    public void exit ()
    {
        // Save the current series.
        this.save();
        // Let the panels save their settings.
        m_dirTreePanel.saveSettings(m_settings);
        m_propertyPanel.saveSettings(m_settings);
        m_dirContentsPanel.saveSettings(m_settings);
        // Set the window position of the main frame.
        boolean bIsMaximized = (this.getState() & MAXIMIZED_BOTH)
            == MAXIMIZED_BOTH;
        if (bIsMaximized)
            this.setState(NORMAL);
        m_settings.setRect(MAIN_WIN_POS, this.getBounds());
        m_settings.setBoolean(MAIN_WIN_MAXIMIZED, bIsMaximized);
        // Set the splitter positions.
        m_settings.setInt(MAIN_WIN_DIR_SPLIT_POS,
            dirSplitPane.getDividerLocation());
        m_settings.setInt(MAIN_WIN_PROP_SPLIT_POS,
            propSplitPane.getDividerLocation());
        // Save the prepared settings.
        m_settings.save();
        // Exit.
        System.exit(0);
    } // exit

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        dirSplitPane = new javax.swing.JSplitPane();
        propSplitPane = new javax.swing.JSplitPane();
        dirPanel = new javax.swing.JPanel();
        propertyPanel = new javax.swing.JPanel();
        contentsPanel = new javax.swing.JPanel();
        statusBar = new javax.swing.JLabel();
        menuBar = new javax.swing.JMenuBar();
        seriesMenu = new javax.swing.JMenu();
        prefsMenuItem = new javax.swing.JMenuItem();
        jSeparator1 = new javax.swing.JSeparator();
        exportMenuItem = new javax.swing.JMenuItem();
        jSeparator2 = new javax.swing.JSeparator();
        exitMenuItem = new javax.swing.JMenuItem();
        viewMenu = new javax.swing.JMenu();
        refreshMenuItem = new javax.swing.JMenuItem();
        helpMenu = new javax.swing.JMenu();

        this.getContentPane().setLayout(new java.awt.GridBagLayout());

        this.setTitle("J Photo-Explorer");
        this.addWindowListener(new java.awt.event.WindowAdapter()
        {
            @Override
			public void windowActivated(java.awt.event.WindowEvent evt)
            {
                ExplorerMainFrame.this.formWindowActivated(evt);
            }
            @Override
			public void windowClosing(java.awt.event.WindowEvent evt)
            {
                ExplorerMainFrame.this.exitForm(evt);
            }
            @Override
			public void windowDeactivated(java.awt.event.WindowEvent evt)
            {
                ExplorerMainFrame.this.formWindowDeactivated(evt);
            }
        });

        dirSplitPane.setDividerLocation(300);
        propSplitPane.setDividerLocation(400);
        propSplitPane.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);
        dirPanel.setLayout(new java.awt.BorderLayout());

        propSplitPane.setLeftComponent(dirPanel);

        propertyPanel.setLayout(new java.awt.BorderLayout());

        propertyPanel.setMinimumSize(new java.awt.Dimension(40, 40));
        propertyPanel.setOpaque(false);
        propertyPanel.setPreferredSize(new java.awt.Dimension(100, 100));
        propSplitPane.setRightComponent(propertyPanel);

        dirSplitPane.setLeftComponent(propSplitPane);

        contentsPanel.setLayout(new java.awt.BorderLayout(0, 6));

        contentsPanel.setMinimumSize(new java.awt.Dimension(200, 224));
        dirSplitPane.setRightComponent(contentsPanel);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        this.getContentPane().add(dirSplitPane, gridBagConstraints);

        statusBar.setText("JPhoto-Explorer");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 4, 2);
        this.getContentPane().add(statusBar, gridBagConstraints);

        seriesMenu.setText("Series");
        prefsMenuItem.setText("Preferences");
        prefsMenuItem.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ExplorerMainFrame.this.prefsMenuItemActionPerformed(evt);
            }
        });

        seriesMenu.add(prefsMenuItem);

        jSeparator1.setForeground(new java.awt.Color(153, 153, 153));
        seriesMenu.add(jSeparator1);

        exportMenuItem.setText("Export ...");
        exportMenuItem.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ExplorerMainFrame.this.exportMenuItemActionPerformed(evt);
            }
        });

        seriesMenu.add(exportMenuItem);

        jSeparator2.setForeground(new java.awt.Color(153, 153, 153));
        seriesMenu.add(jSeparator2);

        exitMenuItem.setText("Exit");
        exitMenuItem.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ExplorerMainFrame.this.exitMenuItemActionPerformed(evt);
            }
        });

        seriesMenu.add(exitMenuItem);

        menuBar.add(seriesMenu);

        viewMenu.setText("View");
        refreshMenuItem.setText("Refresh");
        refreshMenuItem.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ExplorerMainFrame.this.refreshMenuItemActionPerformed(evt);
            }
        });

        viewMenu.add(refreshMenuItem);

        menuBar.add(viewMenu);

        helpMenu.setText("Help");
        menuBar.add(helpMenu);

        this.setJMenuBar(menuBar);

        this.pack();
    }// </editor-fold>//GEN-END:initComponents

    private void formWindowDeactivated(java.awt.event.WindowEvent evt)//GEN-FIRST:event_formWindowDeactivated
    {//GEN-HEADEREND:event_formWindowDeactivated
        m_dirTreePanel.setInactiveColors();
        m_propertyPanel.setInactiveColors();
        m_dirContentsPanel.setInactiveColors();
    }//GEN-LAST:event_formWindowDeactivated

    private void formWindowActivated(java.awt.event.WindowEvent evt)//GEN-FIRST:event_formWindowActivated
    {//GEN-HEADEREND:event_formWindowActivated
        m_dirTreePanel.setActiveColors();
        m_propertyPanel.setActiveColors();
        m_dirContentsPanel.setActiveColors();
    }//GEN-LAST:event_formWindowActivated

    private void exportMenuItemActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_exportMenuItemActionPerformed
    {//GEN-HEADEREND:event_exportMenuItemActionPerformed
        String s = m_settings.get(Settings.EXPORT_DIRECTORY);
        if (s.length() == 0)
        {
            JOptionPane.showMessageDialog(this, "Please specify the Web "
                + "export directory in the Series->Options dialog",
                AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
            return;
        }
        ExportSelDialog dlg = new ExportSelDialog(this, m_seriesContainer,
            m_settings, true);
        if (!dlg.showDialog())
            return;
        // Export the selected group or series.
        DirectoryObject group = dlg.getGroup();
        this.startExport((group != null) ? group : dlg.getSeries(),
            dlg.getExportMain(), dlg.getExportIndex(), dlg.getForceExport());
    }//GEN-LAST:event_exportMenuItemActionPerformed

    private void refreshMenuItemActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_refreshMenuItemActionPerformed
    {//GEN-HEADEREND:event_refreshMenuItemActionPerformed
        this.refresh();
    }//GEN-LAST:event_refreshMenuItemActionPerformed

    /**
     * Shows the dialog to change the settings for the whole directory
     * structure.
     * @param evt - The event
     */
    private void prefsMenuItemActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_prefsMenuItemActionPerformed
    {//GEN-HEADEREND:event_prefsMenuItemActionPerformed
        this.save();
        SettingsDialog dlg = new SettingsDialog
            (this, m_seriesContainer, m_settings, true);
        if (!dlg.showDialog())
            return;
        // Refresh the directory structure that has been changed by the dialog.
        this.refresh();
    }//GEN-LAST:event_prefsMenuItemActionPerformed

    private void aboutMenuItemActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_aboutMenuItemActionPerformed
    {//GEN-HEADEREND:event_aboutMenuItemActionPerformed
        AboutDialog dlg = new AboutDialog(this, true);
        dlg.setVisible(true);
    }//GEN-LAST:event_aboutMenuItemActionPerformed

    private void exitMenuItemActionPerformed(java.awt.event.ActionEvent evt)
    {//GEN-FIRST:event_exitMenuItemActionPerformed
        this.exit();
    }//GEN-LAST:event_exitMenuItemActionPerformed

    /**
     * Exit the <i>J Photo Explorer</i> application.
     * @param evt - The window close event
     */
    private void exitForm(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_exitForm
        this.exit();
    }//GEN-LAST:event_exitForm

    // Variables declaration - do not modify//GEN-BEGIN:variables
    public javax.swing.JPanel contentsPanel;
    public javax.swing.JPanel dirPanel;
    public javax.swing.JSplitPane dirSplitPane;
    public javax.swing.JMenuItem exitMenuItem;
    public javax.swing.JMenuItem exportMenuItem;
    public javax.swing.JMenu helpMenu;
    public javax.swing.JSeparator jSeparator1;
    public javax.swing.JSeparator jSeparator2;
    public javax.swing.JMenuBar menuBar;
    public javax.swing.JMenuItem prefsMenuItem;
    public javax.swing.JSplitPane propSplitPane;
    public javax.swing.JPanel propertyPanel;
    public javax.swing.JMenuItem refreshMenuItem;
    public javax.swing.JMenu seriesMenu;
    public javax.swing.JLabel statusBar;
    public javax.swing.JMenu viewMenu;
    // End of variables declaration//GEN-END:variables
} // ExplorerMainFrame
