/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ImageHolder
    Created: 7 January, 2003
        $Id: ImageHolder.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;

import magick.MagickImage;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.webexport.ImageMagickExporter;

/**
 * Helper class that represents entries in the image load list. Each stands for
 * one image of one size to load.
 */
class ToLoadItem
{
    public FileObject fileObj;
    public String     scaleCategory;
    public int        imgSize;
    public Component  parent;

    public ToLoadItem
        ( FileObject fObj
        , String     sCategory
        , int        iSize
        , Component  par
        )
    {
        this.fileObj = fObj;
        this.scaleCategory = sCategory;
        this.imgSize = iSize;
        this.parent = par;
    } // ToLoadItem
} // ToLoadItem

/**
 * Instances of this class hold the file system object type images and can
 * provide scaled previews for images (GIF, JPEG, and PNG file objects; TIFF
 * images if ImageMagick is available).
 */
public class ImageHolder implements Runnable
{
    public  static final String SC_PROPERTIES = "prop";
    private static final String TN_DIR_NAME   = "_JP_Preview";

    private boolean   m_bImgLoaded;

    private ImageIcon m_imgImg, m_imgHTML, m_imgXML, m_imgPDF, m_imgMsWord;
    private ImageIcon m_imgMsVisio, m_imgText, m_imgCAD, m_imgZIP, m_imgRef;
    private ImageIcon m_imgHTMLRef, m_imgDir, m_imgSep, m_imgImgPrev;
    private ImageIcon m_imgOther;
    private String    m_lastError;

    private Map<String, Map<String, ImageIcon>> m_imageCats;
    private List<String>                        m_transientCats;

    // Image loading threading members.
    private List<ToLoadItem> m_toLoad;
    private Thread           m_loadThread;
    private Boolean          m_bHasNewImages;

    /**
     * Creates a new <tt>ImageHolder</tt> instance.
     */
    public ImageHolder ()
    {
        // Initialize the image variables.
        m_imgImg = m_imgHTML = m_imgXML = m_imgPDF = m_imgMsWord = null;
        m_imgMsVisio = m_imgText = m_imgCAD = m_imgZIP = m_imgRef = null;
        m_imgHTMLRef = m_imgDir = m_imgSep = m_imgImgPrev = m_imgOther = null;
        m_bImgLoaded = false;
        m_imageCats = new HashMap<String, Map<String, ImageIcon>>(5);
        m_transientCats = new ArrayList<String>();
        m_lastError = "";
        // Initialize the image loading threading members.
        m_toLoad = new ArrayList<ToLoadItem>(10);
        m_loadThread = null;
        m_bHasNewImages = new Boolean(false);
    } // ImageHolder

    /**
     * Returns the last error string.
     * @return Error string
     */
    public String getLastError ()
    {
        String retStr = m_lastError;
        m_lastError = "";
        return retStr;
    } // getLastError

    /**
     * Loads a standard image from our JAR file.
     * @param name - File name of the image file
     * @param parent - Parent component for the media tracker
     * @return Image icon representing the loaded image
     */
    public static ImageIcon loadStandardImage (String name, Component parent)
    {
        Toolkit tk = Toolkit.getDefaultToolkit();
        URL imgURL = ClassLoader.getSystemResource(name);
        if (imgURL == null)
            return null;
        MediaTracker track = new MediaTracker(parent);
        Image img = tk.getImage(imgURL);
        track.addImage(img, 0);
        try
        {
            track.waitForAll();
        }
        catch (InterruptedException iexc)
        {
        }
        return new ImageIcon(img);
    } // loadStandardImage

    /**
     * Marks a scale category as transient, i. e., cached images can be
     * flushed.
     * @param scaleCategory - Identifier describing the image size
     */
    public void addTransientScaleCategory (String scaleCategory)
    {
        if (!m_transientCats.contains(scaleCategory))
        m_transientCats.add(scaleCategory);
    } // addTransientScaleCategory

    private static final int THOP_RENAME = 1;
    private static final int THOP_DELETE = 2;

    /**
     * Private helper of the thumbnail renaming and deletion methods that
     * carries out the operation.
     * @param fsObj - File system object that still has its old name
     * @param newName - New file name
     * @param op - Operation identifier
     */
    public void thumbnailsOp
        ( AbstractFSObject fsObj
        , String           newName
        , int              op
        )
    {
        File       parentDir, f, newFile;
        String[]   dirContents;
        String     oldBaseName, newBaseName, namePattern, imgFilePath;
        FileObject fileObj;
        int        fileType, i, index;

        // Check arguments and prepare.
        if (!(fsObj instanceof FileObject))
            return;
        fileObj = (FileObject) fsObj;
        fileType = fileObj.getFileType();
        if (fileType != FileObject.TYPE_IMAGE_PREVIEW &&
            fileType != FileObject.TYPE_IMAGE)
        {
            return;
        }
        imgFilePath = fileObj.getFullPath();
        // Get the parent directory of the thumbnail files of this file object.
        parentDir = this.getThumbnailDirectory(fileObj);
        if (parentDir == null)
            return;
        // Get the base thumbnail file names.
        oldBaseName = fileObj.getBaseFileName();
        newBaseName = "";
        if (op == THOP_RENAME)
        {
            index = newName.lastIndexOf(".");
            if (index > 0)
                newBaseName = newName.substring(0, index);
            else
                newBaseName = newName;
        }
        namePattern = "^_" + oldBaseName + "_[A-Za-z0-9]+\\.png$";
        // Search for the thumbnail files.
        dirContents = parentDir.list();
        if (dirContents != null)
        {
            for (i = 0; i < dirContents.length; i++)
            {
                if (dirContents[i].matches(namePattern))
                {
                    f = new File(parentDir, dirContents[i]);
                    if (!f.isFile())
                        continue;
                    switch (op)
                    {
                    case THOP_RENAME:
                        index = dirContents[i].lastIndexOf("_");
                        newFile = new File
                            (parentDir, "_" + newBaseName +
                             dirContents[i].substring(index));
                        if (!f.renameTo(newFile))
                        {
                            System.err.println
                                ("W: Cannot rename thumbnail file from:\n   "
                                + f.getAbsolutePath()
                                + "\nto:\n   " + newFile.getAbsolutePath());
                        }
                        break;
                    case THOP_DELETE:
                        if (!f.delete())
                        {
                            System.err.println("W: Cannot delete thumbnail file "
                                + f);
                        }
                        break;
                    }
                }
            }
        }
        // Remove the changed files from the thumbnails cache.
        if (m_imageCats == null)
            return;
        Map<String, ImageIcon> imageMap;
        Iterator<Map<String, ImageIcon>> catMapIter = m_imageCats.values().iterator();
        while (catMapIter.hasNext())
        {
            imageMap = catMapIter.next();
            imageMap.remove(imgFilePath);
        }
    } // thumbnailsOp

    /**
     * Renames the thumbnail files when file object <tt>fsObj</tt> is about to
     * be renamed to <tt>newName</tt>.
     * @param fsObj - File system object that still has its old name
     * @param newName - New file name
     */
    public void renameThumbnails
        ( AbstractFSObject fsObj
        , String           newName
        )
    {
        this.thumbnailsOp(fsObj, newName, THOP_RENAME);
    } // renameThumbnails

    /**
     * Deletes the thumbnail files when file object <tt>fsObj</tt> is about to
     * be deleted.
     * @param fsObj - File object
     */
    public void deleteThumbnails
        ( AbstractFSObject fsObj
        )
    {
        this.thumbnailsOp(fsObj, null, THOP_DELETE);
    } // deleteThumbnails

    /**
     * Moves the thumbnail files in a directory under a new parent directory.
     * @param dirObj - Directory whose files and subdirectories should be
     * moved
     * @param newParentObj - New parent directory
     * @return <tt>True</tt> iff successful
     */
    public boolean moveThumbnailDirUnder
        ( DirectoryObject dirObj
        , DirectoryObject newParentObj
        )
    {
        File oldDir = this.getThumbnailDirectory(dirObj);
        File newParentDir = this.getThumbnailDirectory(newParentObj);
        if (oldDir == null || newParentDir == null)
            return false;
        if (oldDir.renameTo(new File(newParentDir, dirObj.getFileName())))
            return false;
        m_imageCats.clear();
        m_imageCats = new HashMap<String, Map<String, ImageIcon>>(5);
        return true;
    } // moveThumbnailDirUnder

    /**
     * Private helper that gets the parent directory of the thumbnail files of
     * a given file object.
     * @param fsObj - File object
     * @return Directory object of the thumbnail files
     */
    private File getThumbnailDirectory (AbstractFSObject fsObj)
    {
        ArrayList<String>       parents;
        DirectoryObject seriesDir;
        File            tnDir;

        parents = new ArrayList<String>(10);
        if (fsObj instanceof DirectoryObject)
            seriesDir = (DirectoryObject) fsObj;
        else
            seriesDir = (DirectoryObject) fsObj.getParent();
        if (seriesDir == null)
            return null;
        while (seriesDir.getParent() != null)
        {
            parents.add(0, seriesDir.getFileName());
            seriesDir = (DirectoryObject) seriesDir.getParent();
        }
        // The thumbnail directory as directory starting parallel to the series
        // directory.
        tnDir = new File(seriesDir.getFullPath() + File.separator + ".."
            + File.separator + TN_DIR_NAME);
        if (!tnDir.isDirectory())
        {
            if (!tnDir.mkdir())
                return null;
        }
        tnDir = new File(tnDir, seriesDir.getFileName().toLowerCase());
        if (!tnDir.isDirectory() && !tnDir.mkdir())
            return null;
        Iterator<String> iter = parents.iterator();
        while (iter.hasNext())
        {
            tnDir = new File(tnDir, iter.next());
            if (!tnDir.isDirectory() && !tnDir.mkdir())
                return null;
        }
        return tnDir;
    } // getThumbnailDirectory

    /**
     * Private helper that gets the file object of a thumbnail file for
     * "fileObj" and the scale category. The scale category is an image size
     * identifier.
     * @param fileObj - File system object whose thumbnail image is needed.
     * @param scaleCategory - Scaling category (identifier of the image size)
     * @return File containing the image or <tt>null</tt> if non-existent
     */
    private File getThumbnailFile
        ( FileObject fileObj
        , String     scaleCategory
        )
    {
        // Get the thumbnail directory.
        File tnDir = this.getThumbnailDirectory(fileObj);
        if (tnDir == null)
            return null;
        // Build the file in the thumbnail directory.
        File tnFile = new File(tnDir,
            '_' + fileObj.getBaseFileName() + '_' + scaleCategory + ".png");
        return tnFile;
    } // getThumbnailFile

    /**
     * Private helper to read an image.
     * @param fileObj - File system object whose image is needed
     * @param scaleCategory - Scaling category (identifier of the image size)
     * @param imgSize - Image size
     * @param parent - Parent component for the media tracker
     * @return Image icon representing the image
     */
    private ImageIcon readScaledImage
        ( FileObject fileObj
        , String     scaleCategory
        , int        imgSize
        , Component  parent
        )
    {
        File          thumbNail;
        MediaTracker  track;
        Map<String, ImageIcon>           imageMap;
        String        imgFilePath;
        Image         scaledImg;
        BufferedImage img;
        ImageIcon     imgIcon;
        int           imgWidth, imgHeight, newWidth, newHeight;

        m_lastError = "";
        imageMap = m_imageCats.get(scaleCategory);
        imgFilePath = fileObj.getFullPath();
        img = null;
        // Try to use ImageMagick because this is much faster.
        try
        {
            ImageMagickExporter ex = new ImageMagickExporter();
            thumbNail = this.getThumbnailFile(fileObj, scaleCategory);
            if (ex.isAvailable() && thumbNail != null)
            {
                // Read the image with dimension.
                MagickImage origMImage = ex.readImage(imgFilePath);
                Dimension siz = origMImage.getDimension();
                if (siz.width > siz.height)
                {
                    newWidth = imgSize;
                    newHeight = (siz.height * newWidth) / siz.width;
                }
                else
                {
                    newHeight = imgSize;
                    newWidth = (siz.width * newHeight) / siz.height;
                }
                // Write the scaled image as thumbnail file.
                ex.writeScaledImage
                    (origMImage, new Dimension(newWidth, newHeight),
                    thumbNail.getAbsolutePath());
                origMImage = null;
                Runtime.getRuntime().gc();
                // Read the image, put it into the map, and return.
                img = ImageIO.read(thumbNail);
                imgIcon = new ImageIcon(img);
                imageMap.put(imgFilePath, imgIcon);
                return imgIcon;
            }
        }
        catch (Exception e)
        {
            System.err.println("W: Exception while reading image for:\n"
                + imgFilePath + "\n" + e);
            e.printStackTrace();
        }
        // Read the image from the file, now using all-Java.
        try
        {
            img = ImageIO.read(new File(imgFilePath));
        }
        catch (OutOfMemoryError me)
        {
            // Panic - no memory.
            thumbNail = null;
            img = null;
            Runtime.getRuntime().gc();
            imageMap.clear();
            Runtime.getRuntime().gc();
            m_lastError = "Out of memory while reading image:\n" + imgFilePath;
            System.err.println("E: " + m_lastError);
            return null;
        }
        catch (Exception exc)
        {
        }
        if (img == null)
            return null;
        // Do we have to resize the image.
        if (imgSize <= 0)
        {
            imgIcon = new ImageIcon(img);
            imageMap.put(imgFilePath, imgIcon);
            return imgIcon;
        }
        // Resize the image.
        imgWidth = img.getWidth(null);
        imgHeight = img.getHeight(null);
        if (imgWidth > imgHeight)
        {
            newWidth = imgSize;
            newHeight = (imgHeight * newWidth) / imgWidth;
        }
        else
        {
            newHeight = imgSize;
            newWidth = (imgWidth * newHeight) / imgHeight;
        }
        try
        {
            scaledImg = img.getScaledInstance
                (newWidth, newHeight, Image.SCALE_AREA_AVERAGING);
            track = new MediaTracker(parent);
            track.addImage(scaledImg, 0);
            try
            {
                track.waitForAll();
            }
            catch (InterruptedException iexc)
            {
            }
        }
        catch (OutOfMemoryError me1)
        {
            // Panic - no memory.
            thumbNail = null;
            img = null;
            scaledImg = null;
            Runtime.getRuntime().gc();
            imageMap.clear();
            Runtime.getRuntime().gc();
            m_lastError = "Out of memory while scaling image:\n" + imgFilePath;
            System.err.println("E: " + m_lastError);
            return null;
        }
        // Write the thumbnail file for this image.
        thumbNail = this.getThumbnailFile(fileObj, scaleCategory);
        if (thumbNail != null)
        {
            BufferedImage scaledBufImg;
            try
            {
                int imgType = img.getType();
                if (imgType <= 0)
                    imgType = BufferedImage.TYPE_INT_ARGB;
                scaledBufImg = new BufferedImage(newWidth, newHeight, imgType);
                Graphics2D gr = scaledBufImg.createGraphics();
                boolean bReady = gr.drawImage(scaledImg, 0, 0, null);
                if (!bReady)
                {
                    // Pfui Teufel (English: Hack).
                    try
                    {
                        Thread.sleep(55);
                    }
                    catch (Exception sleepEx)
                    {
                    }
                }
                ImageIO.write(scaledBufImg, "png", thumbNail);
                scaledBufImg.flush();
                scaledBufImg = null;
            }
            catch (Exception exc)
            {
                m_lastError = "While writing thumbnail \"" + thumbNail
                    + "\"\nException: " + exc;
                System.err.println("E: " + m_lastError);
            }
        }
        // Try to get rid of the original image.
        img.flush();
        img = null;
        Runtime.getRuntime().gc();
        // Return the resized image.
        imgIcon = new ImageIcon(scaledImg);
        imageMap.put(imgFilePath, imgIcon);
        return imgIcon;
    } // readScaledImage

    /**
     * Private method to add an image to the list of images to be loaded in
     * another thread.
     * @param fileObj - File system object whose image is needed
     * @param scaleCategory - Scaling category (identifier of the image size)
     * @param imgSize - Image size
     * @param parent - Parent component for the media tracker
     */
    private void addToLoad
        ( FileObject fileObj
        , String     scaleCategory
        , int        imgSize
        , Component  parent
        )
    {
        synchronized (m_toLoad)
        {
            // Check whether the item is already in the list.
            Iterator<ToLoadItem> iter = m_toLoad.iterator();
            ToLoadItem item;
            while (iter.hasNext())
            {
                item = iter.next();
                if (item.fileObj == fileObj &&
                    item.scaleCategory.equals(scaleCategory))
                {
                    return;
                }
            }
            // Add the item to load.
            m_toLoad.add(new ToLoadItem(fileObj, scaleCategory, imgSize,
                parent));
        }
        if (m_loadThread == null)
        {
            m_loadThread = new Thread(this);
            m_loadThread.setPriority(Thread.MIN_PRIORITY);
            m_loadThread.start();
        }
    } // addToLoad

    /**
     * Tests whether there are new images.
     * @return <tt>True</tt> if there are new images
     */
    public boolean hasNewImages ()
    {
        synchronized (m_bHasNewImages)
        {
            return m_bHasNewImages.booleanValue();
        }
    } // hasNewImages

    /**
     * Tests whether all requested images have been loaded.
     * @return <tt>True</tt> if all images have been loaded
     */
    public boolean hasAllImages ()
    {
        boolean bHasAll = (m_loadThread == null);
        if (bHasAll)
        {
            synchronized (m_bHasNewImages)
            {
                m_bHasNewImages = new Boolean(false);
            }
        }
        return bHasAll;
    } // hasAllImages

    /** Stops the thread that loads images in the background. */
    public void stopBackgroundImageLoading ()
    {
        synchronized (m_toLoad)
        {
            m_toLoad.clear();
        }
        synchronized (m_bHasNewImages)
        {
            m_bHasNewImages = new Boolean(false);
        }
    } // stopBackgroundImageLoading

    /**
     * Runs to load images in the background. The information on the images to
     * load is in the to-do list.
     */
    public void run ()
    {
        ToLoadItem item;
        ImageIcon  image;
        int        outOfMemoryCount = 0;
        int        sleepInterval = 2222;

        item = null;
        for (;;)
        {
            // Check the to-do list.
            if (item == null)
            {
                synchronized (m_toLoad)
                {
                    if (m_toLoad.size() == 0)
                       break;
                    item = m_toLoad.remove(0);
                    outOfMemoryCount = 0;
                    sleepInterval = 2222;
                }
            }
            else if (outOfMemoryCount > 5)
            {
                // We have got 5 out-of-memory's for the same image - let it be
                synchronized (m_toLoad)
                {
                    System.err.println("E: Too many out-of-memory errors");
                    m_toLoad.clear();
                    break;
                }
            }
            // Load the image.
            try
            {
                image = this.readScaledImage(item.fileObj, item.scaleCategory,
                    item.imgSize, item.parent);
            }
            catch (OutOfMemoryError me)
            {
                image = null;
            }
            catch (Exception exc)
            {
                image = null;

            }
            // If we have no memory - good night.
            if (image == null)
            {
                outOfMemoryCount++;
                try
                {
                    Runtime.getRuntime().gc();
                    sleepInterval += outOfMemoryCount * 777;
                    Thread.sleep(sleepInterval);
                    Runtime.getRuntime().gc();
                }
                catch (InterruptedException ie)
                {
                }
                // We process the item again.
                continue;
            }
            item = null;
            synchronized (m_bHasNewImages)
            {
                m_bHasNewImages = new Boolean(true);
            }
        }
        m_loadThread = null;
    } // run

    /**
     * Private helper to load a preview image for a file object.
     * @param fileObj - File system object whose image is needed
     * @param scaleCategory - Scaling category (identifier of the image size)
     * @param imgSize - Image size
     * @param parent - Parent component for the media tracker
     * @param bWaitForImage - <tt>True</tt> to wait for the image
     * @return Image icon representing the image
     */
    private ImageIcon loadImage
        ( FileObject fileObj
        , String     scaleCategory
        , int        imgSize
        , Component  parent
        , boolean    bWaitForImage
        )
    {
        File          thumbNail;
        Map<String, ImageIcon>           imageMap;
        String        imgFilePath;
        BufferedImage img;
        ImageIcon     imgIcon;

        m_lastError = "";
        // Do we have the scaled image in the image maps?
        imageMap = m_imageCats.get(scaleCategory);
        if (imageMap == null)
        {
            imageMap = new HashMap<String, ImageIcon>(300);
            m_imageCats.put(scaleCategory, imageMap);
        }
        imgFilePath = fileObj.getFullPath();
        Object imgObj = imageMap.get(imgFilePath);
        if (imgObj != null && imgObj instanceof ImageIcon)
            return (ImageIcon) imgObj;
        // Flush the images as needed.
        if (m_transientCats.contains(scaleCategory) &&
            imageMap.size() > 600)
        {
            imageMap.clear();
        }
        img = null;
        // If there is a thumbnail image for this file and scale category,
        // return the thumbnail.
        thumbNail = this.getThumbnailFile(fileObj, scaleCategory);
        if (thumbNail != null && thumbNail.isFile())
        {
            try
            {
                // Is the thumbnail newer as the image ?-
                long imgTime = (new File(imgFilePath)).lastModified();
                long tnTime = thumbNail.lastModified();
                if (tnTime >= imgTime)
                    img = ImageIO.read(thumbNail);
            }
            catch (Exception exc)
            {
            }
            if (img != null)
            {
                imgIcon = new ImageIcon(img);
                imageMap.put(imgFilePath, imgIcon);
                return imgIcon;
            }
        }
        if (bWaitForImage)
        {
            return this.readScaledImage(fileObj, scaleCategory, imgSize,
                parent);
        }
        this.addToLoad(fileObj, scaleCategory, imgSize, parent);
        return null;
    } // loadImage

    /**
     * Gets the type image for a file system object.
     * @param fsObj - The file system object
     * @param bWithImagePreview - <tt>True</tt> to load the preview image of an
     * image file object
     * @param scaleCategory - Identifier for an image size category
     * @param imgSize - Image size
     * @param parent - Parent component for the media tracker
     * @param bWaitForImage - <tt>True</tt> to wait for the image to be loaded,
     * <tt>false</tt> to load it in a background thread
     * @return ImageIcon
     */
    public ImageIcon getTypeImage
        ( AbstractFSObject fsObj
        , boolean          bWithImagePreview
        , String           scaleCategory
        , int              imgSize
        , Component        parent
        , boolean          bWaitForImage
        )
    {
        // Load the images.
        if (!m_bImgLoaded)
        {
            m_imgImg = loadStandardImage("images/type-img.gif", parent);
            m_imgImgPrev = loadStandardImage
                ("images/type-img-preview.gif", parent);
            m_imgHTML = loadStandardImage("images/type-html.gif",parent);
            m_imgXML = loadStandardImage("images/type-xml.gif", parent);
            m_imgText = loadStandardImage("images/type-text.gif", parent);
            m_imgCAD = loadStandardImage("images/type-cad.gif", parent);
            m_imgZIP = loadStandardImage("images/type-zip.gif", parent);
            m_imgPDF = loadStandardImage("images/type-pdf.gif", parent);
            m_imgMsWord = loadStandardImage("images/type-msword.gif", parent);
            m_imgMsVisio = loadStandardImage("images/type-msvisio.gif", parent);
            m_imgRef = loadStandardImage("images/type-ref.gif", parent);
            m_imgHTMLRef = loadStandardImage
                ("images/type-html-ref.gif", parent);
            m_imgDir = loadStandardImage("images/type-dir.gif", parent);
            m_imgSep = loadStandardImage("images/type-sep.gif", parent);
            m_imgOther = loadStandardImage("images/type-other.gif", parent);
            m_bImgLoaded = true;
        }
        // Check the file entry.
        if (fsObj instanceof FileObject)
        {
            switch (((FileObject) fsObj).getFileType())
            {
                case FileObject.TYPE_IMAGE_PREVIEW:
                    if (bWithImagePreview)
                    {
                        ImageIcon imgIcon = this.loadImage((FileObject) fsObj,
                            scaleCategory, imgSize, parent, bWaitForImage);
                        if (imgIcon != null)
                            return imgIcon;
                    }
                    // Otherwise, return the image icon.
                    return m_imgImgPrev;
                case FileObject.TYPE_IMAGE:
                    return m_imgImg;
                case FileObject.TYPE_HTML_DOC:
                    return m_imgHTML;
                case FileObject.TYPE_XML_DOC:
                    return m_imgXML;
                case FileObject.TYPE_PDF_DOC:
                    return m_imgPDF;
                case FileObject.TYPE_MSWORD_DOC:
                    return m_imgMsWord;
                case FileObject.TYPE_MSVISIO_DOC:
                    return m_imgMsVisio;
                case FileObject.TYPE_TEXT_DOC:
                    return m_imgText;
                case FileObject.TYPE_CAD_DOC:
                    return m_imgCAD;
                case FileObject.TYPE_ZIP_DOC:
                    return m_imgZIP;
                case FileObject.TYPE_REFERENCE:
                    return m_imgRef;
                case FileObject.TYPE_HTML_REF:
                    return m_imgHTMLRef;
                case FileObject.TYPE_SEPARATOR:
                    return m_imgSep;
            }
        }
        else if (fsObj instanceof DirectoryObject)
            return m_imgDir;
        return m_imgOther;
    } // getTypeImage
} // ImageHolder
