/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: HtmlViewDialog
    Created: 2 January, 2003
        $Id: HtmlViewDialog.java 41 2006-01-14 15:49:02Z Dirk $
  $Revision: 41 $
      $Date: 2006-01-14 16:49:02 +0100 (Sa, 14 Jan 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JOptionPane;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLFrameHyperlinkEvent;

import com.dgrossmann.photo.AppInfo;

/**
 * The <i>HTML file view</i> dialog class.
 */
public class HtmlViewDialog extends JDialog
    implements HyperlinkListener
{
    private URL m_homeURL;
    private URL m_lastURL;

    /**
     * Creates a new <tt>HtmlViewDialog</tt> form instance.
     * @param parent - Parent frame in the user interface
     * @param title - Dialog title
     * @param htmlFilePath - Full path of the HTML file to be shown
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public HtmlViewDialog
        ( Frame   parent
        , String  title
        , String  htmlFilePath
        , boolean modal
        )
    {
        super(parent, modal);
        this.initialize(title, htmlFilePath);
    } // HtmlViewDialog

    /**
     * Creates a new <tt>HtmlViewDialog</tt> form instance.
     * @param parent - Parent dialog in the user interface
     * @param title - Dialog title
     * @param htmlFilePath - Full path of the HTML file to be shown
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public HtmlViewDialog
        ( Dialog  parent
        , String  title
        , String  htmlFilePath
        , boolean modal
        )
    {
        super(parent, modal);
        this.initialize(title, htmlFilePath);
    } // HtmlViewDialog

    /**
     * Private methode to initialize this instance.
     * @param title - The window title
     * @param htmlFilePath - The path to the HTML file to display
     */
    private void initialize (String  title, String  htmlFilePath)
    {
        this.initComponents();
        // Set the contents.
        String tit = (title != null) ? title : " ";
        if (tit.length() > 1)
            tit += " - ";
        tit += "HTML View";
        this.setTitle(tit);
        titleLabel.setText(htmlFilePath);
        backButton.setEnabled(false);
        // Set the listener.
        editorPane.addHyperlinkListener(this);
        // Set the HTML file.
        m_homeURL = m_lastURL = null;
        try
        {
            m_homeURL = new URL("file:///" + htmlFilePath);
            editorPane.setPage(m_homeURL);
        }
        catch (MalformedURLException e)
        {
            JOptionPane.showMessageDialog
                (this, "Wrong file URL: " + e.toString(), AppInfo.APP_NAME,
                JOptionPane.ERROR_MESSAGE);
        }
        catch (IOException ioExc)
        {
            JOptionPane.showMessageDialog
                (this, "I/O Error: " + ioExc.toString(), AppInfo.APP_NAME,
                JOptionPane.ERROR_MESSAGE);
        }
        // Set the dialog position.
        Toolkit   tk = Toolkit.getDefaultToolkit();
        Dimension siz = tk.getScreenSize();
        Rectangle frameRect = new Rectangle();
        frameRect.width = 950;
        if (frameRect.width >= siz.width)
            frameRect.width = siz.width;
        frameRect.height = siz.height - 34;
        frameRect.x = 0;
        frameRect.y = 0;
        this.setBounds(frameRect);
    } // initialize

    /**
     * Hyperlink listener method.
     * @see javax.swing.event.HyperlinkListener#hyperlinkUpdate(javax.swing.event.HyperlinkEvent)
     */
    public void hyperlinkUpdate(HyperlinkEvent e)
    {
        backButton.setEnabled(true);
        if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED)
        {
            JEditorPane pane = (JEditorPane) e.getSource();
            m_lastURL = pane.getPage();
            if (e instanceof HTMLFrameHyperlinkEvent)
            {
                HTMLFrameHyperlinkEvent evt = (HTMLFrameHyperlinkEvent) e;
                HTMLDocument doc = (HTMLDocument) pane.getDocument();
                doc.processHTMLFrameHyperlinkEvent(evt);
            }
            else
            {
                try
                {
                    pane.setPage(e.getURL());
                }
                catch (Throwable t)
                {
                    JOptionPane.showMessageDialog(this,
                        "Cannot find URL:\n" + e.getURL(),
                        AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
                }
            }
        }
    } // hyperlinkUpdate

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents()//GEN-BEGIN:initComponents
    {
        titleLabel = new javax.swing.JLabel();
        jScrollPane1 = new javax.swing.JScrollPane();
        editorPane = new javax.swing.JEditorPane();
        buttonPanel = new javax.swing.JPanel();
        homeButton = new javax.swing.JButton();
        backButton = new javax.swing.JButton();
        closeButton = new javax.swing.JButton();

        addWindowListener(new java.awt.event.WindowAdapter()
        {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                closeDialog(evt);
            }
        });

        titleLabel.setText(" ");
        getContentPane().add(titleLabel, java.awt.BorderLayout.NORTH);

        editorPane.setEditable(false);
        editorPane.setContentType("text/html");
        jScrollPane1.setViewportView(editorPane);

        getContentPane().add(jScrollPane1, java.awt.BorderLayout.CENTER);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        homeButton.setText("Home");
        homeButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                homeButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(homeButton);

        backButton.setText("Back");
        backButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                backButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(backButton);

        closeButton.setText("Close");
        closeButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                closeButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(closeButton);

        getContentPane().add(buttonPanel, java.awt.BorderLayout.SOUTH);

        pack();
    }//GEN-END:initComponents

    private void backButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_backButtonActionPerformed
    {//GEN-HEADEREND:event_backButtonActionPerformed
        backButton.setEnabled(false);
        if (m_lastURL == null)
            return;
        try
        {
            editorPane.setPage(m_lastURL);
        }
        catch (IOException ioExc)
        {
            System.err.println("E: IO Exception: " + ioExc.toString());
        }
        m_lastURL = null;
    }//GEN-LAST:event_backButtonActionPerformed

    private void homeButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_homeButtonActionPerformed
    {//GEN-HEADEREND:event_homeButtonActionPerformed
        backButton.setEnabled(false);
        m_lastURL = null;
        if (m_homeURL == null)
            return;
        try
        {
            editorPane.setPage(m_homeURL);
        }
        catch (IOException ioExc)
        {
            System.err.println("E: IO Exception: " + ioExc.toString());
        }
    }//GEN-LAST:event_homeButtonActionPerformed

    private void closeButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_closeButtonActionPerformed
    {//GEN-HEADEREND:event_closeButtonActionPerformed
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeButtonActionPerformed
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeDialog
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel titleLabel;
    private javax.swing.JButton backButton;
    private javax.swing.JEditorPane editorPane;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JButton homeButton;
    private javax.swing.JButton closeButton;
    // End of variables declaration//GEN-END:variables
    
} // HtmlViewDialog
