/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ImageViewDialog
    Created: 6 January, 2003
        $Id: ImageViewDialog.java 58 2006-02-21 18:58:22Z Dirk $
  $Revision: 58 $
      $Date: 2006-02-21 19:58:22 +0100 (Di, 21 Feb 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JDialog;

import magick.MagickImage;
import magick.MagickProducer;

import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.webexport.ImageMagickExporter;

/**
 * The <i>image view</i> dialog class that shows an image.
 * @author Dirk Grossmann
 */
public class ImageViewDialog extends JDialog
{
    private Color               m_errorColor;
    private FileObject          m_fileObj;
    private BufferedImage       m_image;
    private Image               m_scaledImage;
    private MagickImage         m_MImage, m_scaledMImage;
    private ImageIcon           m_imageIcon, m_scaledImageIcon;
    private Dimension           m_imgSize, m_scaledSize;
    private ImageMagickExporter m_magick;

    /**
     * Creates a new <tt>ImageViewDialog form instance.
     * @param fileObj - File object representing the image to be shown
     * @param parent - Parent dialog
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public ImageViewDialog
        ( FileObject fileObj
        , Dialog     parent
        , boolean    modal
        )
    {
        super(parent, modal);
        parent.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        this.initialize(fileObj);
        parent.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    } // ImageViewDialog

    /**
     * Creates a new <tt>ImageViewDialog form instance.
     * @param fileObj - File object representing the image to be shown
     * @param parent - Parent frame
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public ImageViewDialog
        ( FileObject fileObj
        , Frame      parent
        , boolean    modal
        )
    {
        super(parent, modal);
        parent.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        this.initialize(fileObj);
        parent.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    } // ImageViewDialog

    /**
     * Private method to initialize this dialog instance.
     * @param fileObj - File object representing the image to be shown
     */
    private void initialize (FileObject fileObj)
    {
        String str;

        m_fileObj = fileObj;
        m_imgSize = new Dimension(0, 0);
        m_scaledSize = new Dimension(0, 0);
        m_image = null;
        m_scaledImage = null;
        m_MImage = m_scaledMImage = null;
        m_imageIcon = m_scaledImageIcon = null;
        m_errorColor = new Color(153, 51, 51);
        // ImageMagick needs more memory here because the image is stored twice
        // while painting, and we get more out-of-memory errors.  Therefore, we
        // use it only if we must.
        m_magick = null;
        if (m_fileObj.isImageMagickRequiredForDisplay() &&
            ImageMagickExporter.isImageMagickAvailable())
        {
            m_magick = new ImageMagickExporter();
        }
        this.initComponents();
        // Set the dialog title.
        str = m_fileObj.getTitlePlain();
        if (str.length() > 0)
            str += " - ";
        str += "Image View";
        this.setTitle(str);
        titleLabel.setText(m_fileObj.getFullPath());
        // Try to load the image.
        try
        {
            this.getOriginalSizeImage();
            // Set the image.
            if (m_imageIcon != null)
            {
                str = " " + m_imgSize.width + " x " + m_imgSize.height
                    + " pixels";
            }
            else
                str = "";
        }
        catch (OutOfMemoryError me)
        {
            // Panic - no memory.
            m_image = null;
            m_scaledImage = null;
            m_MImage = m_scaledMImage = null;
            m_imageIcon = m_scaledImageIcon = null;
            Runtime.getRuntime().gc();
            str = "Out of memory while reading image - Please close the window.";
            sizeLabel.setForeground(m_errorColor);
            // Disable the zoom buttons.
            fitToWindowButton.setEnabled(false);
            origSizeButton.setEnabled(false);
            Runtime.getRuntime().gc();
        }
        catch (Exception e)
        {
            str = "Cannot load original image: " + e.getMessage();
            sizeLabel.setForeground(m_errorColor);
            fitToWindowButton.setEnabled(false);
            origSizeButton.setEnabled(false);
        }
        sizeLabel.setText(str);
        // Set the dialog position.
        Toolkit tk = Toolkit.getDefaultToolkit();
        Dimension siz = tk.getScreenSize();
        Rectangle frameRect = new Rectangle();
        int newWidth;
        if (m_imgSize.width > 0 && m_imgSize.width < 2500)
            newWidth = (m_imgSize.width * siz.height) / m_imgSize.height - 60;
        else
            newWidth = siz.width;
        frameRect.width = (newWidth < siz.width) ? newWidth : siz.width;
        frameRect.height = siz.height;
        frameRect.x = 0;
        frameRect.y = 0;
        this.setBounds(frameRect);
        // Set the resized image.
        if (m_imgSize.width > 2500 || m_imgSize.height > 2500)
        {
            // We cannot afford loading the scaled image.
            fitToWindowButton.setEnabled(false);
            this.origSizeButtonActionPerformed(null);
            str += " (big image - resize buttons disabled)";
            sizeLabel.setText(str);
        }
        else
        {
            this.fitToWindowButtonActionPerformed(null);
            Runtime.getRuntime().gc();
        }
    } // initialize

    /**
     * Private helper method to read an image.
     * @return The image in its original size
     * @throws Exception
     */
    private ImageIcon getOriginalSizeImage () throws Exception
    {
        if (m_imageIcon != null)
            return m_imageIcon;
        if (m_magick != null)
        {
            try
            {
                // Read the image using ImageMagick.
                m_MImage = m_magick.readImage(m_fileObj.getFullPath());
                m_imgSize = m_MImage.getDimension();
                Image img = this.createImage(new MagickProducer(m_MImage));
                MediaTracker track = new MediaTracker(this);
                track.addImage(img, 0);
                try
                {
                    track.waitForAll();
                }
                catch (InterruptedException iexc)
                {
                }
                m_imageIcon = new ImageIcon(img);
                return m_imageIcon;
            }
            catch (Exception ex)
            {
                System.err.println("E: JM: Cannot read image:");
                ex.printStackTrace();
            }
        }
        // Now try it the Java way.
        m_image = ImageIO.read(new File(m_fileObj.getFullPath()));
        m_imgSize.width = m_image.getWidth();
        m_imgSize.height = m_image.getHeight();
        m_imageIcon = new ImageIcon(m_image);
        return m_imageIcon;
    } // getOriginalSizeImage

    /**
     * Private helper to get a resized image.
     * @return The scaled image
     * @throws Exception
     */
    private ImageIcon getScaledImage () throws Exception
    {
        MediaTracker track;

        if (m_scaledImageIcon != null)
            return m_scaledImageIcon;
        if (m_magick != null)
        {
            try
            {
                m_scaledMImage = m_MImage.scaleImage
                    (m_scaledSize.width, m_scaledSize.height);
                Image img = this.createImage
                    (new MagickProducer(m_scaledMImage));
                track = new MediaTracker(this);
                track.addImage(img, 0);
                try
                {
                    track.waitForAll();
                }
                catch (InterruptedException iexc)
                {
                }
                m_scaledImageIcon = new ImageIcon(img);
                return m_scaledImageIcon;
            }
            catch (Exception ex)
            {
                System.err.println("E: JM: Cannot resize image to ("
                    + m_scaledSize.width + " x " + m_scaledSize.height + ")");
                ex.printStackTrace();
                // This is the hard way.
                m_magick = null;
                m_MImage = null;
                m_imageIcon = null;
                this.getOriginalSizeImage();
            }
        }
        // Now get the scaled image the Java way.
        if (m_image == null)
            return null;
        m_scaledImage = m_image.getScaledInstance(m_scaledSize.width,
            m_scaledSize.height, Image.SCALE_AREA_AVERAGING);
        track = new MediaTracker(this);
        track.addImage(m_scaledImage, 0);
        try
        {
            track.waitForAll();
        }
        catch (InterruptedException iexc)
        {
        }
        m_scaledImageIcon = new ImageIcon(m_scaledImage);
        return m_scaledImageIcon;
    } // getScaledImage

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents()//GEN-BEGIN:initComponents
    {
        titleLabel = new javax.swing.JLabel();
        westSpacer = new javax.swing.JLabel();
        eastSpacer = new javax.swing.JLabel();
        imgViewPanel = new javax.swing.JPanel();
        imgViewScrollPane = new javax.swing.JScrollPane();
        imageLabel = new javax.swing.JLabel();
        bottomPanel = new javax.swing.JPanel();
        sizeLabel = new javax.swing.JLabel();
        buttonPanel = new javax.swing.JPanel();
        fitToWindowButton = new javax.swing.JButton();
        origSizeButton = new javax.swing.JButton();
        sepLabel = new javax.swing.JLabel();
        closeButton = new javax.swing.JButton();

        addWindowListener(new java.awt.event.WindowAdapter()
        {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                closeDialog(evt);
            }
        });

        titleLabel.setText(" ");
        getContentPane().add(titleLabel, java.awt.BorderLayout.NORTH);

        westSpacer.setText(" ");
        getContentPane().add(westSpacer, java.awt.BorderLayout.WEST);

        eastSpacer.setText(" ");
        getContentPane().add(eastSpacer, java.awt.BorderLayout.EAST);

        imgViewPanel.setLayout(new java.awt.BorderLayout());

        imgViewScrollPane.setMinimumSize(new java.awt.Dimension(55, 55));
        imgViewScrollPane.setPreferredSize(new java.awt.Dimension(1300, 1300));
        imgViewScrollPane.setViewportView(imageLabel);

        imgViewPanel.add(imgViewScrollPane, java.awt.BorderLayout.CENTER);

        getContentPane().add(imgViewPanel, java.awt.BorderLayout.CENTER);

        bottomPanel.setLayout(new java.awt.BorderLayout());

        sizeLabel.setText("3000 x 3000 pixels");
        bottomPanel.add(sizeLabel, java.awt.BorderLayout.WEST);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        fitToWindowButton.setText("Fit to Window");
        fitToWindowButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                fitToWindowButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(fitToWindowButton);

        origSizeButton.setText("Original Size");
        origSizeButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                origSizeButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(origSizeButton);

        sepLabel.setText(" ");
        buttonPanel.add(sepLabel);

        closeButton.setText("Close");
        closeButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                closeButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(closeButton);

        bottomPanel.add(buttonPanel, java.awt.BorderLayout.CENTER);

        getContentPane().add(bottomPanel, java.awt.BorderLayout.SOUTH);

        pack();
    }//GEN-END:initComponents

    private void fitToWindowButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_fitToWindowButtonActionPerformed
    {//GEN-HEADEREND:event_fitToWindowButtonActionPerformed
        Dimension viewSize;
        int       newHeight;

        if (m_magick == null && m_image == null ||
            m_magick != null && m_MImage == null)
        {
            return;
        }
        try
        {
            viewSize = imgViewScrollPane.getSize();
            viewSize.width -= 10;
            viewSize.height -= 20;
            if (viewSize.width >= m_imgSize.width &&
                viewSize.height >= m_imgSize.height)
            {
                this.origSizeButtonActionPerformed(evt);
                return;
            }
            newHeight = (m_imgSize.height * viewSize.width) / m_imgSize.width;
            if (newHeight > viewSize.height)
            {
                viewSize.width = (m_imgSize.width * viewSize.height) /
                    m_imgSize.height;
            }
            else
            {
                viewSize.height = (m_imgSize.height * viewSize.width) /
                m_imgSize.width;
            }
            // Set the scaled image into the image label.
            if (!m_scaledSize.equals(viewSize) || m_scaledImageIcon == null)
            {
                m_scaledSize = viewSize;
                m_scaledImage = null;
                m_scaledMImage = null;
                m_scaledImageIcon = null;
                Runtime.getRuntime().gc();
                // Get the scaled image.
                this.getScaledImage();
            }
            if (m_scaledImageIcon != null)
                imageLabel.setIcon(m_scaledImageIcon);
            else
                imageLabel.setText("Scaled image not available");
        }
        catch (OutOfMemoryError me)
        {
            // Panic - no memory.
            Runtime.getRuntime().gc();
            sizeLabel.setForeground(m_errorColor);
            sizeLabel.setText("Out of memory while scaling image - "
                + "Please close the window.");
            // Disable the zoom buttons.
            fitToWindowButton.setEnabled(false);
            origSizeButton.setEnabled(false);
            this.origSizeButtonActionPerformed(evt);
            Runtime.getRuntime().gc();
        }
        catch (Exception e)
        {
            String str = "Cannot load scaled image: " + e.getMessage();
            e.printStackTrace();
            sizeLabel.setText(str);
            sizeLabel.setForeground(m_errorColor);
            fitToWindowButton.setEnabled(false);
        }
    }//GEN-LAST:event_fitToWindowButtonActionPerformed

    private void origSizeButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_origSizeButtonActionPerformed
    {//GEN-HEADEREND:event_origSizeButtonActionPerformed
        try
        {
            if (m_imageIcon != null)
            {
                imageLabel.setIcon(m_imageIcon);
                // Maximize the dialog.
                Dimension siz = Toolkit.getDefaultToolkit().getScreenSize();
                Rectangle frameRect = this.getBounds();
                if (frameRect.width < siz.width)
                {
                    frameRect.width = siz.width;
                    frameRect.x = 0;
                    this.setBounds(frameRect);
                    this.validate();
                }
            }
            else
                imageLabel.setText("Image not available");
        }
        catch (OutOfMemoryError me)
        {
            // Panic - no memory.
            Runtime.getRuntime().gc();
            sizeLabel.setForeground(m_errorColor);
            sizeLabel.setText("Out of memory - Please close the window.");
            // Disable the zoom buttons.
            fitToWindowButton.setEnabled(false);
            origSizeButton.setEnabled(false);
        }
    }//GEN-LAST:event_origSizeButtonActionPerformed

    private void closeButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_closeButtonActionPerformed
    {//GEN-HEADEREND:event_closeButtonActionPerformed
        this.setVisible(false);
        this.dispose();
        // Free the resources for the images.
        m_imgSize = new Dimension(0, 0);
        m_scaledSize = new Dimension(0, 0);
        m_image = null;
        m_scaledImage = null;
        m_MImage = m_scaledMImage = null;
        m_imageIcon = m_scaledImageIcon = null;
        Runtime.getRuntime().gc();
    }//GEN-LAST:event_closeButtonActionPerformed
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
        // Free the resources for the images.
        m_imgSize = new Dimension(0, 0);
        m_scaledSize = new Dimension(0, 0);
        m_image = null;
        m_scaledImage = null;
        m_MImage = m_scaledMImage = null;
        m_imageIcon = m_scaledImageIcon = null;
        Runtime.getRuntime().gc();
    }//GEN-LAST:event_closeDialog
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane imgViewScrollPane;
    private javax.swing.JLabel titleLabel;
    private javax.swing.JLabel westSpacer;
    private javax.swing.JLabel sepLabel;
    private javax.swing.JButton origSizeButton;
    private javax.swing.JLabel sizeLabel;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JPanel imgViewPanel;
    private javax.swing.JPanel bottomPanel;
    private javax.swing.JLabel imageLabel;
    private javax.swing.JButton fitToWindowButton;
    private javax.swing.JButton closeButton;
    private javax.swing.JLabel eastSpacer;
    // End of variables declaration//GEN-END:variables

} // ImageViewDialog
