/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ContentsTableModel
    Created: 2 January, 2003
        $Id: ContentsTableModel.java 174 2010-02-03 18:33:07Z dirk $
  $Revision: 174 $
      $Date: 2010-02-03 19:33:07 +0100 (Mi, 03 Feb 2010) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel.contents;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.JTree;
import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.ui.ImageHolder;
import com.dgrossmann.photo.ui.panel.DirTreeModel;
import com.dgrossmann.photo.webexport.ExportException;
import com.dgrossmann.photo.webexport.ExportFactory;
import com.dgrossmann.photo.webexport.IWebExport;

/**
 * Helper class for the image load thread. It re-selects the selected files in
 * the table.
 * @author Dirk Grossmann
 */
class ImageTimerTask implements ActionListener
{
    private ContentsTableModel m_model;
    private ImageHolder        m_imgHolder;

    /**
     * Creates a new <tt>ImageTimerTask</tt> instance.
     * @param model - The table model of the contents table
     * @param imgHolder - The image holder
     */
    public ImageTimerTask (ContentsTableModel model, ImageHolder imgHolder)
    {
        m_model = model;
        m_imgHolder = imgHolder;
    } // ImageTimerTask

    /**
     * Called when the timer expires.
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed (ActionEvent evt)
    {
        if (m_imgHolder.hasNewImages())
        {
            List<AbstractFSObject> fileObjs = null;
            if (m_model.m_parentComponent != null)
                fileObjs = m_model.m_parentPanel.getSelectedFiles();
            m_model.fireTableDataChanged();
            if (m_model.m_parentComponent != null && fileObjs != null &&
                !fileObjs.isEmpty())
            {
                m_model.m_parentPanel.selectCurrentFiles(false, fileObjs);
            }
        }
        if (m_imgHolder.hasAllImages() && m_model.m_timer != null)
            m_model.m_timer.stop();
    } // actionPerformed
} // ImageTimerTask

/**
 * Implements the table model for the file list in the contents panel.
 * @author Dirk Grossmann
 */
public class ContentsTableModel extends AbstractTableModel
{
    // Hard-coded headings and column types.
    private static final String[] s_columnNames =
    {
        "Image", "Name", "Title", "Modified", "Export"
    };
    private static final Class<?>[] s_columnTypes =
    {
        ImageIcon.class, String.class, String.class, String.class,
        Boolean.class
    };

    public    Component         m_parentComponent;
    public    ContentsPanel     m_parentPanel;
    public    javax.swing.Timer m_timer;

    protected JTree             m_dirTree;
    protected DirTreeModel      m_dirTreeModel;
    protected ImageHolder       m_imgHolder;
    protected DirectoryObject   m_currentDir;
    protected boolean           m_bWithImagePreview;

    /**
     * Creates a new <tt>ContentsTableModel</tt> instance.
     * @param parent - Parent UI component
     * @param frame - Parent panel in the user interface; can be <tt>null</tt>
     * @param imgHolder - Image holder object
     */
    public ContentsTableModel
        ( Component     parent
        , ContentsPanel frame
        , ImageHolder   imgHolder
        )
    {
        m_currentDir = null;
        m_parentComponent = parent;
        m_parentPanel = frame;
        m_imgHolder = imgHolder;
        m_dirTree = null;
        m_dirTreeModel = null;
        m_bWithImagePreview = true;
        m_timer = null;
    } // ContentsTableModel

    /**
     * Sets the directory tree component.
     * @param dirTree - The tree
     */
    public void setDirTree (JTree dirTree)
    {
        m_dirTree = dirTree;
        if (m_dirTree != null)
            m_dirTreeModel = (DirTreeModel) m_dirTree.getModel();
        else
            m_dirTreeModel = null;
    } // setDirTree

    /**
     * Sets the current directory object.
     * @param currentDir - Current directory object
     */
    public void setCurrentDir (DirectoryObject currentDir)
    {
        m_currentDir = currentDir;
        // Stop image loading.
        m_imgHolder.stopBackgroundImageLoading();
        if (m_timer != null)
            m_timer.stop();
        // Fire the change event.
        TableModelEvent evt = new TableModelEvent(this);
        this.fireTableChanged(evt);
    } // setCurrentDir

    /** Refreshes the table. */
    public void refresh ()
    {
        TableModelEvent evt = new TableModelEvent(this);
        this.fireTableChanged(evt);
    } // refresh

    /**
     * Gets the column count.
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount ()
    {
        return s_columnNames.length;
    } // getColumnCount

    /**
     * Gets the name of the column identified by index.
     * @see javax.swing.table.AbstractTableModel#getColumnName(int)
     */
    @Override
	public String getColumnName (int index)
    {
        return s_columnNames[index];
    } // getColumnName

    /**
     * Gets the class of the column identified by index.
     * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
     */
    @Override
	public Class<?> getColumnClass (int index)
    {
        return s_columnTypes[index];
    } // getColumnClass

    /**
     * Gets the type image.
     * @param fsObj - The file system object
     * @return The icon representing its type
     */
    public ImageIcon getTypeImage (AbstractFSObject fsObj)
    {
        ImageIcon icon = m_imgHolder.getTypeImage(fsObj, m_bWithImagePreview,
        	"list", 50, m_parentComponent, false);
        if (m_bWithImagePreview && !m_imgHolder.hasAllImages())
        {
            // We have got a standard icon and the actual image is loading.
            if (m_timer == null)
            {
                m_timer = new javax.swing.Timer(1000,
                    new ImageTimerTask(this, m_imgHolder));
                m_timer.start();
            }
            else if (!m_timer.isRunning())
                m_timer.start();
        }
        return icon;
    } // getTypeImage

    /**
     * Gets the row count.
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount ()
    {
        return (m_currentDir != null) ?
            m_currentDir.getSubDirCount() + m_currentDir.getFileCount()
            : 0;
    } // getRowCount

    /**
     * Gets the directory or file object at the specified <tt>0</tt>-based
     * row index in the table.
     * @param rowIndex - <tt>0</tt>-based table row index
     * @return File system object at this index or <tt>null</tt>
     */
    public AbstractFSObject getFSObjectAt (int rowIndex)
    {
        int dirCount;

        if (m_currentDir == null || rowIndex < 0 ||
            rowIndex >= this.getRowCount())
        {
            return null;
        }
        dirCount = m_currentDir.getSubDirCount();
        if (rowIndex < dirCount)
            return m_currentDir.getSubDirAt(rowIndex);
        return m_currentDir.getFileAt(rowIndex - dirCount);
    } // getFSObjectAt

    /**
     * Gets the row index of the specified file system object in the table.
     * @param fsObj - File system object
     * @return Table row index or <tt>-1</tt> if not in the table.
     */
    public int getIndexOfFSObject (AbstractFSObject fsObj)
    {
        int index;

        if (m_currentDir == null)
            return -1;
        if (fsObj instanceof DirectoryObject)
            return m_currentDir.getIndexOfSubDir((DirectoryObject) fsObj);
        if (fsObj instanceof FileObject)
        {
            index = m_currentDir.getIndexOfFile((FileObject) fsObj);
            if (index >= 0)
                return index + m_currentDir.getSubDirCount();
            return -1;
        }
        return -1;
    } // getIndexOfFSObject

    /**
     * Adds a file system object as child to the current directory at the
     * specified <tt>0</tt>-based table row index.
     * @param fsObj - File system object to add
     * @param rowIndex - Row index
     * @return <tt>False</tt> if the table model instance does not have a
     * current directory; <tt>true</tt> otherwise
     */
    public boolean addFSObject (AbstractFSObject fsObj, int rowIndex)
    {
        if (m_currentDir == null)
            return false;
        if (fsObj instanceof FileObject)
            rowIndex -= m_currentDir.getSubDirCount();
        m_currentDir.addChild(fsObj, rowIndex);
        this.refresh();
        return true;
    } // addFSObject

    /**
     * Removes the files system object as child from the current directory.
     * @param fsObj - File system object to remove
     * @return <tt>False</tt> if the table model instance does not have a
     * current directory or <tt>fsObj</tt> is not a file or directory object;
     * <tt>true</tt> otherwise
     */
    public boolean removeFSObject (AbstractFSObject fsObj)
    {
        if (m_currentDir == null)
            return false;
        if (fsObj instanceof FileObject)
        {
            m_currentDir.removeFile((FileObject) fsObj);
            this.refresh();
            return true;
        }
        if (fsObj instanceof DirectoryObject)
        {
            m_currentDir.removeSubDir((DirectoryObject) fsObj);
            if (m_dirTreeModel != null)
                m_dirTreeModel.fireTreeStructureChanged(m_currentDir);
        }
        return false;
    } // removeFSObject

    /**
     * Gets the value at (row, column).
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt (int row, int column)
    {
        if (column < 0 || column >= this.getColumnCount())
            return "";
        AbstractFSObject fsObj = this.getFSObjectAt(row);
        if (fsObj == null)
            return "";
        switch (column)
        {
            case 0: // File type image.
                return this.getTypeImage(fsObj);
            case 1: // File name.
                return fsObj.getFileName();
            case 2: // File title.
                if (fsObj instanceof FileObject &&
                    ((FileObject) fsObj).isSeparator())
                {
                    return "-------------------------------";
                }
                // Return the title with HTML entity references expanded.
                return AbstractFSObject.transformAccents(fsObj.getTitle(true),
                    false);
            case 3: // Modification date/time
                return fsObj.getModDateTimeString(true);
            case 4: // Export?
                return new Boolean(fsObj.isToExport());
        }
        return "";
    } // getValueAt

    /**
     * Gets whether a cell is editable.
     * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
     */
    @Override
	public boolean isCellEditable (int row, int column)
    {
        if (m_currentDir == null || row < 0 || row >= this.getRowCount() ||
            column < 0 || column >= this.getColumnCount())
        {
            return false;
        }
        AbstractFSObject fsObj = this.getFSObjectAt(row);
        if (fsObj == null)
            return false;
        if (fsObj instanceof FileObject && ((FileObject) fsObj).isSeparator())
            return column == 4 /*Export*/;
        if (fsObj.isReference() && (column == 1 /*File Name*/))
            return false;
        return column != 0 /*Preview*/ && column != 3 /*Modified*/;
    } // isCellEditable

    /**
     * Sets the object value at (row, column)
     * @param newVal - New cell value
     * @param row - <tt>0</tt>-based cell row index
     * @param column - <tt>0</tt>-based cell column index
     */
    @Override
	public void setValueAt (Object newVal, int row, int column)
    {
        if (column < 0 || column >= this.getColumnCount() || newVal == null)
            return;
        AbstractFSObject fsObj = this.getFSObjectAt(row);
        if (fsObj == null)
            return;
        String str = newVal.toString().trim();
        switch (column)
        {
            case 1: // File name.
                if (fsObj.isReference() ||
                    fsObj.getFileName().equalsIgnoreCase(str))
                {
                    return;
                }
                // Does a file of the desired name already exist ?-
                if (fsObj.getParent() != null)
                {
                    File f = new File(fsObj.getParent().getFullPath(), str);
                    if (f.exists())
                    {
                        JOptionPane.showMessageDialog(m_parentComponent,
                            "Cannot rename as a file with the following "
                            + "name already exists:\n" + f.getAbsolutePath(),
                            AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                        return;
                    }
                }
                // No, continue with renaming.
                if (m_parentPanel != null)
                {
                    m_imgHolder.renameThumbnails(fsObj, str);
                    m_parentPanel.renameExportedFiles(fsObj, null, str);
                    fsObj.setFileName(str, true);
                }
                if (fsObj instanceof DirectoryObject && m_dirTreeModel != null)
                    m_dirTreeModel.fireTreeStructureChanged(m_currentDir);
                break;
            case 2: // File title.
                if (fsObj instanceof FileObject &&
                    ((FileObject) fsObj).isSeparator())
                {
                    return;
                }
                fsObj.set(AbstractFSObject.TITLE, str);
                break;
            case 4: // To export?
                boolean bVal = false;
                if (newVal instanceof Boolean)
                    bVal = ((Boolean) newVal).booleanValue();
                else
                {
                    if (str.length() > 0)
                    {
                        char ch = str.toLowerCase().charAt(0);
                        if (ch == 'y' || ch == 't' || ch == '1')
                            bVal = true;
                    }
                }
                fsObj.setToExport(bVal);
                // Delete exported files if we set it to false.
                if (!bVal && (fsObj instanceof FileObject))
                {
                	try
					{
						ExportFactory.getExport(
							m_parentPanel.getFrame().getSettings(),
							m_parentPanel.getFrame().getSeriesContainer(),
							m_parentPanel).deleteExportedFiles(fsObj,
								IWebExport.EXPORT_ALL);
					}
					catch (ExportException ignored)
					{
					}
                }
        }
        m_currentDir.setChanged();
        m_parentPanel.firePropertiesChanged(fsObj);
    } // setValueAt
} // ContentsTableModel
