/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ImageMagickExporter
    Created: 9 January, 2003
        $Id: ImageMagickExporter.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.webexport;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.io.File;

import magick.ColorspaceType;
import magick.CompressionType;
import magick.DrawInfo;
import magick.ImageInfo;
import magick.Magick;
import magick.MagickImage;
import magick.PixelPacket;

import com.dgrossmann.photo.dir.FileObject;

/**
 * Exporter that uses ImageMagick (JMagick) to convert the images.
 */
public class ImageMagickExporter extends BaseExporter
{
    private static boolean s_bAvailabilityChecked = false;
    private static boolean s_bAvailable           = false;

    /** Constructor of a new <tt>ImageMagickExporter</tt> instance. */
    public ImageMagickExporter ()
    {
        super();
    } // ImageMagickExporter

    /**
     * Gets a description of the method used by the exporter.
     * @return Description string
     */
    public String getDescription ()
    {
        return "ImageMagick";
    } // getDescription

    /**
     * Returns whether ImageMagick and Java-Magick are available.
     * @return <tt>True</tt> iff all prerequisites are available
     */
    public static boolean isImageMagickAvailable ()
    {
        if (s_bAvailabilityChecked)
            return s_bAvailable;
        s_bAvailabilityChecked = true;
        s_bAvailable = false;
        try
        {
            Rectangle rect = new Rectangle(0, 0, 80, 40);
            Magick.parseImageGeometry("50x50", rect);
            s_bAvailable = true;
            return true;
        }
        catch (NoClassDefFoundError cnf)
        {
            System.err.println("I: ImageMagick (JMagick) class \"Magick\" "
                + "not found");
        }
        catch (UnsatisfiedLinkError ule)
        {
            System.err.println("I: ImageMagick (JMagick) native method "
                + "library not found");
        }
        catch (Exception e)
        {
            System.err.println("I: Exception while accessing ImageMagick "
            + "(JMagick) class \"Magick\":\n" + e);
        }
        return false;
    } // isImageMagickAvailable

    /**
     * Returns whether the exporter has all prerequisites installed (e.g., the
     * ImageMagick exporter requires ImageMagick and Java-Magick).
     * @return <tt>True</tt> iff all prerequisites are available
     */
    public boolean isAvailable ()
    {
        return isImageMagickAvailable();
    } // isAvailable

    /**
     * Reads an image.
     * @param fileName - File name of the image
     * @return MagickImage, Image
     * @throws Exception on error
     */
    public MagickImage readImage (String fileName)
    	throws Exception
    {
        ImageInfo info = new ImageInfo(fileName);
        info.setColorspace(ColorspaceType.RGBColorspace);
        return new MagickImage(info);
    } // readImage

    /**
     * Writes a scaled image.
     * @param origImage - Image to be written
     * @param newSize - New image size
     * @param fileName - File name
     * @throws Exception on error
     */
    public void writeScaledImage
        ( MagickImage origImage
        , Dimension   newSize
        , String      fileName
        ) throws Exception
    {
        MagickImage scaledImage;
        ImageInfo   info;

        if (!origImage.setImageAttribute("filter", "Cubic"))
            System.err.println("E: JM: Cannot set filter attribute");
        scaledImage = origImage.scaleImage(newSize.width, newSize.height);
        scaledImage.setCompression(CompressionType.LZWCompression);
        scaledImage.setFileName(fileName);
        info = new ImageInfo(fileName);
        scaledImage.writeImage(info);
    } // writeScaledImage

    /**
     * Method that really carries out the file export. The <tt>progress</tt>
     * object is filled with the output directory and the file names, and the
     * output directory exists.
     * @param fileObj - File object to export
     * @param progress - Filled progress object
     * @throws ExportException on failure
     */
    @Override
	protected void exportFile
        ( FileObject     fileObj
        , ExportProgress progress
        ) throws ExportException
    {
        MagickImage origImage, scaledImage;
        ImageInfo   info;
        Dimension   size;
        String      copyrightStr, outFileName;
        DrawInfo    drawInfo;

        try
        {
            // Read the original image.
            info = new ImageInfo(fileObj.getFullPath());
            info.setColorspace(ColorspaceType.RGBColorspace);
            origImage = new MagickImage(info);
            // Get the image dimensions.
            size = origImage.getDimension();
            progress.setOriginalSize(size);
            this.fillImageSizes(progress);
            // Scale to the "main" image size.
            if (!origImage.setImageAttribute("filter", "Cubic"))
                System.err.println("E: JM: Cannot set filter attribute");
            if (!progress.isMainUpToDate())
            {
                size = progress.getMainSize();
                outFileName = progress.getOutputDirectory() + File.separator
                    + progress.getMainFileName();
                scaledImage = origImage.scaleImage(size.width, size.height);
                scaledImage.setCompression(CompressionType.JPEGCompression);
                scaledImage.setFileName(outFileName);
                // Print the copyright string.
                copyrightStr = this.getCopyrightString(fileObj);
                if (copyrightStr != null)
                {
                	// Black "shadow" text.
                	drawInfo = new DrawInfo(new ImageInfo());
                	drawInfo.setFill(PixelPacket.queryColorDatabase("black"));
                	drawInfo.setOpacity(0);
                	drawInfo.setPointsize(11);
                	drawInfo.setFont("Arial");
                	drawInfo.setTextAntialias(false);
                	drawInfo.setText(copyrightStr);
                	drawInfo.setGeometry("+" + (size.width - 130)
                		+ "+" + (size.height - 2));
                	scaledImage.annotateImage(drawInfo);
                	// White "foreground" text.
                	drawInfo = new DrawInfo(new ImageInfo());
                	drawInfo.setFill(PixelPacket.queryColorDatabase("white"));
                	drawInfo.setOpacity(0);
                	drawInfo.setPointsize(11);
                	drawInfo.setFont("Arial");
                	drawInfo.setTextAntialias(false);
                	drawInfo.setText(copyrightStr);
                	drawInfo.setGeometry("+" + (size.width - 131)
                		+ "+" + (size.height - 3));
                	scaledImage.annotateImage(drawInfo);
                }
                // Write the main image.
                info = new ImageInfo(outFileName);
                info.setQuality(this.getConversionQuality(fileObj, true));
                scaledImage.writeImage(info);
            }
            // Scale to the "overview" image size.
            if (!progress.isIndexUpToDate())
            {
                size = progress.getIndexSize();
                outFileName = progress.getOutputDirectory() + File.separator
                    + progress.getIndexFileName();
                scaledImage = origImage.scaleImage(size.width, size.height);
                scaledImage.setCompression(CompressionType.JPEGCompression);
                scaledImage.setFileName(outFileName);
                info = new ImageInfo(outFileName);
                info.setQuality(this.getConversionQuality(fileObj, false));
                scaledImage.writeImage(info);
            }
        }
        catch (Exception e)
        {
            ExportException exc = new ExportException(fileObj, e);
            exc.log();
            throw exc;
        }
    } // exportFile
} // ImageMagickExporter
